/**
 * External dependencies
 */
import React, { Fragment } from 'react';
import classNames from 'classnames';

/**
 * WordPress dependencies
 */
const { __, sprintf } = wp.i18n;

/**
 * Internal dependencies
 */
import './wizard.scss';
import Box from '../../components/sui-box';
import { getLink } from '../../../js/utils/helpers';
import Icon from '../../components/sui-icon';
import ProgressBar from '../../components/sui-progress';
import Tag from '../../components/sui-tag';
import Toggle from '../../components/sui-toggle';
import SettingsRow from '../../components/sui-box-settings/row';
import Button from '../../components/sui-button';
import Tabs from '../../components/sui-tabs';
import Tooltip from '../../components/sui-tooltip';
import { createInterpolateElement } from '@wordpress/element';
import ToggleSettingsRow from './ToggleSettingsRow';
import RadioImageOption from './RadioImageOption';
import PerformanceCard from './PerformanceCard';
import ProcessSetupConfiguration from '../../../js/scanners/ProcessSetupConfiguration';
import ProcessingConfigurationModal from './ProcessingConfigurationModal';
import ConnectNotice from '../../components/connect-notice';
import NotificationsSettings from '../../components/notifications-settings';
import UpsellCard from './UpsellCard';

/**
 * Wizard module, extends React.Component.
 *
 * @since 3.3.1
 */
export default class Wizard extends React.Component {
	/**
	 * Component constructor.
	 *
	 * @param {Object} props
	 */
	constructor( props ) {
		super( props );

		this.state = {
			steps: {
				1: __( 'Getting Started', 'wphb' ),
				2: __( 'Customize Features', 'wphb' ),
				3: __( 'Setup Summary', 'wphb' ),
				4: __( 'Reports & Alerts', 'wphb' )
			},
			scanning: false,
			skip: {
				advCacheFile: false,
			}
		};

		this.continueToNextStep = this.continueToNextStep.bind( this );
	}

	handleProcessSetupFinish() {
		this.setState( { scanning: false } );
		this.props.getPerformanceScore( false );
	}

	/**
	 * Cancel the scanning process
	 */
	cancelScanning() {
		if ( this.processConfiguration ) {
			this.processConfiguration.cancel();
			this.processConfiguration = null;
		}

		this.setState( { scanning: false } );
	}

	/**
	 * Run actions on component update.
	 *
	 * @param {Object} prevProps
	 * @param {Object} prevState
	 */
	componentDidUpdate( prevProps, prevState ) {
		if ( 1 === this.props.step && this.props.showConflicts ) {
			if ( this.props.step === prevProps.step && this.props.showConflicts === prevProps.showConflicts ) {
				return; // Nothing changed after re-checking status.
			}

			// We need to save our state, so we don't show extra stuff on next step.
			this.setState( {
				skip: {
					advCacheFile: ! this.props.issues.advCacheFile
				}
			} );

			jQuery( '.sui-box-header' ).on( 'click', this.toggleContent );
		}

		if ( true === this.state.scanning && 2 === this.props.step ) {
			this.processConfiguration = new ProcessSetupConfiguration( 100, 0, { ...this.props, onProcessSetupFinish: this.handleProcessSetupFinish.bind( this ) } );
			this.processConfiguration.start();
		}

		if ( 3 <= this.props.step && this.state.scanning ) {
			this.setState( { scanning: false } );
		}
	}

	/**
	 * Show/hide content block with issues.
	 *
	 * @param {Object} e
	 */
	toggleContent( e ) {
		e.currentTarget.parentNode.classList.toggle( 'open' );
	}

	/**
	 * Get navigation.
	 *
	 * @return {JSX.Element} Side navigation
	 */
	getNavigation() {
		const mobileSteps = Object.entries( this.state.steps ).map( ( step, key ) => {
			if ( 4 === key ) {
				return null;
			}

			const x1 = key * 20;
			const x2 = step[ 0 ] * 20;
			const stroke = this.props.step <= step[ 0 ] ? '#D8D8D8' : '#1ABC9C';

			return <line key={ key } x1={ x1 + '%' } x2={ x2 + '%' } className="line-mobile" stroke={ stroke } />;
		} );

		const steps = Object.entries( this.state.steps ).map( ( step, key ) => {
			const classes = classNames( {
				current: parseInt( step[ 0 ] ) === this.props.step,
				done: parseInt( step[ 0 ] ) < this.props.step,
			} );

			return (
				<React.Fragment key={ key }>
					<li className={ classes }>
						{ step[ 1 ] }
						{ 'done' !== classes && <span>{ step[ 0 ] }</span> }
						{ 'done' === classes && <Icon classes="sui-icon-check" /> }
						{ 3 > key && <svg focusable="false" aria-hidden="true"><line y1="0" y2="20px" /></svg> }
					</li>
				</React.Fragment>
			);
		} );

		return (
			<div className="sui-sidenav wizard-sidenav__nav">
				<svg focusable="false" aria-hidden="true">
					{ mobileSteps }
				</svg>
				<ul>{ steps }</ul>
				<div className="nav-menu-actions">
					{ ( ( 1 === this.props.step && ! this.props.showConflicts ) || ( 1 !== this.props.step || ( this.props.showConflicts && ! this.props.issues.advCacheFile ) ) ) && 4 !== this.props.step &&
						<Button
							onClick={ this.continueToNextStep }
							type="button"
							classes={ [ 'sui-button', 'wphb-btn-black' ] }
							text={ __( 'Continue', 'wphb' ) } /> }
					{ 1 === this.props.step && this.props.showConflicts && this.props.issues.advCacheFile &&
						<Button
							onClick={ this.props.skipConflicts }
							type="button"
							classes={ [ 'sui-button', 'wphb-btn-black' ] }
							text={ __( 'Continue anyway', 'wphb' ) } /> }
					{ 4 === this.props.step &&
						<Button
							onClick={ this.props.finish }
							type="button"
							classes={ [ 'sui-button', 'wphb-btn-black' ] }
							text={ __( 'Save & Exit', 'wphb' ) } /> }
					{ 1 !== this.props.step &&
					<Button
						onClick={ this.props.prevStep }
						type="button"
						classes={ [ 'sui-button', 'sui-button-ghost' ] }
						text={ __( 'Back', 'wphb' ) } /> }
				</div>
				<div className="sidenav-footer">
					<span className="sui-description">
						<Button
							onClick={ this.props.quitWizard }
							text={ __( 'Skip Wizard', 'wphb' ) }
							classes={ [ 'skip-wizard', 'sui-button-icon-right' ] }
							icon="sui-icon-close" />
					</span>
				</div>
			</div>
		);
	}

	/**
	 * Wizard header.
	 *
	 * @return {JSX.Element} Header block
	 */
	getHeader() {
		let title = this.state.steps[ this.props.step ];
		let name = this.state.steps[ this.props.step ].replace( /\s+/g, '-' ).toLowerCase();

		if ( 1 === this.props.step && this.props.showConflicts ) {
			name = ! this.props.issues.advCacheFile ? 'success' : 'failed';
			title = __( 'Plugin Conflict', 'wphb' );
		} else if ( 6 === this.props.step ) {
			name = 'success';
			title = __( 'Wizard Completed!', 'wphb' );
		}

		return (
			<React.Fragment>
				<img
					className="sui-image"
					style={ 1 === this.props.step ? { maxWidth: '40px' } : {} }
					alt={ this.state.steps[ this.props.step ] }
					src={ getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/' + name + '.png' }
					srcSet={
						getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/' + name + '.png 1x, ' +
						getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/' + name + '@2x.png 2x'
					} />
				{ 1 !== this.props.step && <h2>{ title }</h2> }
			</React.Fragment>
		);
	}

	/**
	 * Plugins compatibility content.
	 *
	 * @return {JSX.Element} Content block.
	 */
	getCompatPluginsContent() {
		if ( ! this.props.showConflicts || this.state.skip.advCacheFile ) {
			return null;
		}

		let title = __( 'No other caching plugin is detected', 'wphb' );
		let icon = 'check-tick sui-success';
		let description = (
			<p className="sui-description">
				{ __( 'No other caching plugin is detected. You can proceed with the setup.', 'wphb' ) }
			</p>
		);

		if ( this.props.issues.advCacheFile ) {
			title = __( 'Another caching plugin is detected', 'wphb' );
			icon = 'warning-alert sui-error';

			const message = createInterpolateElement(
				__( 'Hummingbird has detected an advanced-cache.php file in your site’s wp-content directory. <a>Manage your plugins</a> and disable any other active caching plugins to ensure Hummingbird’s page caching works properly.', 'wphb' ),
				{
					a: <a href={ getLink( 'plugins' ) } />
				}
			);

			description = (
				<React.Fragment>
					<p className="sui-description">{ message }</p>
					<p className="sui-description">
						{ __( 'If no other caching plugins are active, the advanced-cache.php may have been left by a previously used caching plugin. You can remove the file from the wp-content directory, or remove it via your file manager or FTP.', 'wphb' ) }
					</p>
				</React.Fragment>
			);
		}

		return (
			<Box
				boxClass="open"
				icon={ icon }
				title={ title }
				headerActions={
					<div className="sui-actions-right">
						<Button
							onClick={ this.toggleContent }
							type="button"
							classes="sui-button-icon"
							icon="sui-icon-chevron-up" />
					</div>
				}
				content={ description }
				footerActions={
					<React.Fragment>
						<Button
							onClick={ this.props.reCheckRequirements }
							type="button"
							classes={ [ 'sui-button', 'sui-button-ghost' ] }
							icon="sui-icon-update"
							text={ __( 'Re-check status', 'wphb' ) } />

						{ this.props.issues.advCacheFile &&
							<div className="sui-actions-right">
								<Button
									onClick={ this.props.removeAdvancedCache }
									type="button"
									classes={ [ 'sui-button', 'sui-button-blue' ] }
									text={ __( 'Remove file', 'wphb' ) } />
							</div> }
					</React.Fragment>
				}
			/>
		);
	}

	/**
	 * Toggle module buttons.
	 *
	 * @return {JSX.Element} Buttons.
	 */
	toggleButtons() {
		if ( 2 !== this.props.step || ! this.props.settings.isFastCGISupported ) {
			return null;
		}

		const sideTabs = [
			{
				title: __( 'Static Server', 'wphb' ),
				checked: this.props.settings.isFastCGISupported && this.props.settings.fastCGI,
				id: 'ssc',
				onClick: () => this.props.toggleModule( 'enable', 'fastCGI' )
			},
			{
				title: __( 'Local Page', 'wphb' ),
				checked: ! ( this.props.settings.isFastCGISupported && this.props.settings.fastCGI ),
				id: 'enable',
				onClick: () => this.props.toggleModule( 'enable', 'cacheEnable' )
			},
		];

		return (
			<div className="wizard-toggle-caching-buttons">
				<div className="wizard-toggle-caching-buttons__title">{ __( 'Caching', 'wphb' ) }</div>
				<Tabs sideTabs="true" menu={ sideTabs } />
			</div>
		);
	}

	/**
	 * Hummingbird Setup Wizard header section
	 *
	 * @return {JSX.Element} Header section.
	 */

	getHeaderSection( {
		iconClass,
		title,
		description
	} ) {
		return (
			<div className="sui-box-settings-row sui-flushed wizard-settings__header">
				<div className="wizard-icon-box">
					<Icon classes={ iconClass } />
				</div>
				<div className="wizard-settings__header--title">
					<h1>{ title }</h1>
					<h4>{ description }</h4>
				</div>
			</div>
		);
	}

	/**
	 * Get customization settings tab.
	 *
	 * @return {JSX.Element} Tab content.
	 */
	getCustomizeSettings() {
		if ( true === this.state.scanning && 2 === this.props.step ) {
			SUI.openModal(
				'wphb-processing-configuration-modal',
				'wpbody-content',
				'wphb-processing-configuration-modal',
			);
		}

		return (
			<Fragment>
				{ this.getHeaderSection( {
					iconClass: 'sui-icon-align-left',
					title: __( 'Customize Features', 'wphb' ),
					description: __( 'Customize your setup to get the best performance for your site. Easily manage asset optimization, CDN, page caching, advanced tools, and performance insights—all in one place.', 'wphb' )
				} ) }
				<div className="sui-box-settings-row sui-flushed wizard-settings__section">
					<div className="wizard-settings__section-content">
						<div className="wizard-settings__section-heading">
							<h2>{ __( 'Asset Optimization', 'wphb' ) }</h2>
						</div>
						<div className="wizard-settings__two-col">
							{ this.props.isNetworkAdmin && (
								<Fragment>
									<ToggleSettingsRow
										text={ __( 'Enable AO on Subsites', 'wphb' ) }
										id="aoEnable"
										checked={ this.props.settings.aoEnable }
										onChange={ this.props.updateSettings }
										icon="sui-icon-smush"
										tooltipText={ __( 'Speed up your site by compressing and minifying JavaScript and CSS files.', 'wphb' ) }
									/>
								</Fragment>
							) }
							{ ! this.props.isNetworkAdmin && (
								<Fragment>
									<ToggleSettingsRow
										text={ __( 'Compress files & clean code', 'wphb' ) }
										id="aoCompress"
										checked={ this.props.settings.aoCompress }
										onChange={ this.props.updateSettings }
										icon="sui-icon-smush"
										tooltipText={ __( 'Speed up your site by compressing and minifying JavaScript and CSS files.', 'wphb' ) }
									/>
									<ToggleSettingsRow
										text={ __( 'Combine JavaScript & CSS', 'wphb' ) }
										id="aoCombine"
										checked={ this.props.settings.aoCombine }
										onChange={ this.props.updateSettings }
										icon="sui-icon-layers"
										tooltipText={ __( 'Merge CSS and JS into fewer files to reduce requests and load pages faster.', 'wphb' ) }
									/>
									<ToggleSettingsRow
										text={ __( 'Swap web fonts', 'wphb' ) }
										id="fontSwap"
										checked={ this.props.settings.fontSwap }
										onChange={ this.props.updateSettings }
										icon="sui-icon-defer"
										tooltipText={ __( 'Make text readable sooner by showing a fallback font while the main font loads.', 'wphb' ) }
									/>
								</Fragment>
							) }
							{ ! this.props.isNetworkAdmin && this.props.isMember && (
								<Fragment>
									<ToggleSettingsRow
										text={ __( 'Delay JavaScript Execution', 'wphb' ) }
										id="delayJS"
										checked={ this.props.settings.delayJS }
										onChange={ this.props.updateSettings }
										icon="sui-icon-clock"
										tooltipText={ __( 'Improve performance by delaying non‑critical JavaScript until user interaction.', 'wphb' ) }
									/>
									<ToggleSettingsRow
										text={ __( 'Generate Critical CSS', 'wphb' ) }
										id="criticalCSS"
										checked={ this.props.settings.criticalCSS }
										onChange={ this.props.updateSettings }
										icon="hum-icon-optimization"
										tooltipText={ __( 'Generate critical CSS automatically to show visible content sooner.', 'wphb' ) }
									/>
								</Fragment>
							) }
						</div>
					</div>
				</div>

				{ this.props.isMember && (
					<div className="sui-box-settings-row sui-flushed wizard-settings__section">
						<div className="wizard-settings__section-content">
							<div className="wizard-settings__section-heading">
								<h2>{ __( 'CDN', 'wphb' ) }</h2>
							</div>
							<div className="wizard-settings__two-col">
								<RadioImageOption
									id="aoCdn-single"
									name="aoCdn"
									value="single"
									label={ __( 'Disable CDN', 'wphb' ) }
									imageSrc={ getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/cdn-local.png' }
									imageSrcSet={ getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/cdn-local@2x.png 2x' }
									imageAlt={ __( 'Disable CDN illustration', 'wphb' ) }
									description={ __( 'Serve all assets from your hosting server without using the CDN', 'wphb' ) }
									checked={ ! this.props.settings.aoCdn }
									onChange={ this.props.updateSettings }
								/>
								<RadioImageOption
									id="aoCdn-global"
									name="aoCdn"
									value="global"
									label={ __( 'Enable CDN', 'wphb' ) }
									imageSrc={ getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/cdn-global.png' }
									imageSrcSet={ getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/cdn-global@2x.png 2x' }
									imageAlt={ __( 'Enable CDN illustration', 'wphb' ) }
									description={ __( 'Serve assets from our global CDN powered by 119 locations worldwide', 'wphb' ) }
									checked={ this.props.settings.aoCdn }
									onChange={ this.props.updateSettings }
								/>
							</div>
						</div>
					</div>
				) }

				<div className="sui-box-settings-row sui-flushed wizard-settings__section">
					<div className="wizard-settings__section-content">
						<div className="wizard-settings__section-heading">
							<h2>{ this.props.settings.isFastCGISupported ? __( 'Page Caching', 'wphb' ) : __( 'Local Page Caching', 'wphb' ) }</h2>
							<label htmlFor="cacheEnable" className="sui-toggle" aria-label="On">
								<div className="sui-toggle-label">
									{ this.props.settings.cacheEnable ? __( 'On', 'wphb' ) : __( 'Off', 'wphb' ) }
								</div>
								<input
									type="checkbox"
									id="cacheEnable"
									name="cacheEnable"
									checked={ this.props.settings.cacheEnable }
									onChange={ this.props.updateSettings }
								/>
								<span className="sui-toggle-slider"></span>
							</label>
						</div>
						{ this.props.settings.cacheEnable && (
							<Fragment>
								{ this.toggleButtons() }
								<div className="wizard-settings__two-col">
									<ToggleSettingsRow
										text={ __( 'Preload caching on Homepage', 'wphb' ) }
										id="preloadHomepage"
										checked={ this.props.settings.preloadHomepage }
										onChange={ this.props.updateSettings }
										icon="sui-icon-undo"
										tooltipText={ __( 'Automatically create cached version of your Homepage. This can be resource-intensive operation, so use it only when required.', 'wphb' ) }
									/>
									{
										! this.props.settings.fastCGI &&
											<ToggleSettingsRow
												text={ __( 'Cache on mobile devices', 'wphb' ) }
												id="cacheOnMobile"
												checked={ this.props.settings.cacheOnMobile }
												onChange={ this.props.updateSettings }
												icon="hum-icon-mobile"
												tooltipText={ __( 'Cache pages on mobile devices to deliver faster load times and reduce data usage.', 'wphb' ) }
											/>
									}
									<ToggleSettingsRow
										text={ __( 'Clear cache after new comment', 'wphb' ) }
										id="clearOnComment"
										checked={ this.props.settings.clearOnComment }
										onChange={ this.props.updateSettings }
										icon="hum-icon-add-comment"
										tooltipText={ __( 'Clear the page cache after new comments so visitors see updated content right away.', 'wphb' ) }
									/>
									<ToggleSettingsRow
										text={ __( 'Show clear cache button in admin bar', 'wphb' ) }
										id="clearCacheButton"
										checked={ this.props.settings.clearCacheButton }
										onChange={ this.props.updateSettings }
										icon="sui-icon-eye"
										tooltipText={ __( 'Add a Clear Cache button to the WordPress admin bar to clear all active cache types.', 'wphb' ) }
									/>
								</div>
							</Fragment>
						) }
					</div>
				</div>

				<div className="sui-box-settings-row sui-flushed wizard-settings__section">
					<div className="wizard-settings__section-content">
						<div className="wizard-settings__section-heading">
							<h2>{ __( 'Advanced Tools', 'wphb' ) }</h2>
						</div>
						<div className="wizard-settings__two-col">
							<ToggleSettingsRow
								text={ __( 'Remove query strings from assets', 'wphb' ) }
								id="queryStrings"
								checked={ this.props.settings.queryStrings }
								onChange={ this.props.updateSettings }
								icon="hum-icon-remove-query"
								tooltipText={ __( 'Some servers and CDNs can’t cache assets with query strings. Removing them improves caching and site speed.', 'wphb' ) }
							/>
							<ToggleSettingsRow
								text={ __( 'Remove default Emoji JS & CSS files', 'wphb' ) }
								id="removeEmoji"
								checked={ this.props.settings.removeEmoji }
								onChange={ this.props.updateSettings }
								icon="hum-icon-emoji"
								tooltipText={ __( 'Remove WordPress’s default emoji scripts to cut extra files and improve performance.', 'wphb' ) }
							/>
							<ToggleSettingsRow
								text={ __( 'Lazy load comments', 'wphb' ) }
								id="lazyLoadComments"
								checked={ this.props.settings.lazyLoadComments }
								onChange={ this.props.updateSettings }
								icon="hum-icon-comment"
								tooltipText={ __( 'Load comments only when needed to speed up pages with many comments.', 'wphb' ) }
							/>
							<ToggleSettingsRow
								text={ __( 'Optimize viewport meta tag', 'wphb' ) }
								id="viewPortMeta"
								checked={ this.props.settings.viewPortMeta }
								onChange={ this.props.updateSettings }
								icon="sui-icon-tablet-portrait"
								tooltipText={ __( 'Automatically set the viewport meta tag on your site for improved mobile responsiveness and Core Web Vitals scores.', 'wphb' ) }
							/>
							{ this.props.hasWoo &&
								<ToggleSettingsRow
									text={ __( 'Faster non-shop pages', 'wphb' ) }
									id="cartFragments"
									checked={ this.props.settings.cartFragments }
									onChange={ this.props.updateSettings }
									icon="hum-icon-page"
									tooltipText={ __( 'Disable WooCommerce cart fragments on non-shop pages to reduce AJAX calls and speed up your site.', 'wphb' ) }
								/> }
						</div>
					</div>
				</div>
			</Fragment>
		);
	}

	/**
	 * Reports and notifications settings tab.
	 *
	 * @return {JSX.Element} Notifications settings content.
	 */
	notificationsSettings() {
		return (
			<>
				{ this.getHeaderSection( {
					iconClass: 'sui-icon-align-left hum-icon-flag',
					title: __( 'Report & Alerts', 'wphb' ),
					description: ( this.props.isConnected ) ? __( 'Stay on top of your site’s performance—schedule reports, choose recipients, and set downtime alerts with ease.', 'wphb' ) : __( 'Schedule performance reports and downtime alerts, including scan frequency, alert recipients, and downtime threshold settings.', 'wphb' ),
				} ) }
				<div className="sui-box-settings-row sui-flushed wizard-settings__section setup-notifications">
					{ ! this.props.isConnected && (
						<div style={ { marginBottom: '30px' } } className="connect-notice-wrapper">
							<ConnectNotice
								onConnect={ () => this.props.finish( 'hubConnect' ) }
							/>
						</div>
					) }
					<NotificationsSettings
						settings={ this.props.settings }
						updateSettings={ this.props.updateSettings }
						isConnected={ this.props.isConnected }
						isMember={ this.props.isMember }
					/>
				</div>
			</>
		);
	}

	/**
	 * Results tab.
	 *
	 * @return {JSX.Element} Tab content.
	 */
	showResults() {
		const zeroScoreLayout = this.props.settings.desktopScore <= 0 && this.props.settings.mobileScore <= 0;
		return (
			<Fragment>
				{ this.getHeaderSection( {
					iconClass: 'sui-icon-wand-magic',
					title: __( 'Setup Summary', 'wphb' ),
					description: __( 'Your performance scan is complete. We’ve applied the best optimizations based on your selections—here’s your summary.', 'wphb' )
				} ) }
				{ this.props.settings.desktopScore > 0 && this.props.settings.mobileScore > 0 &&
				<div className="sui-box-settings-row sui-flushed">
					<PerformanceCard desktopScore={ this.props.settings.desktopScore } mobileScore={ this.props.settings.mobileScore } isMember={ this.props.isMember } onTrack={ this.props.trackUpsell } />
				</div>
				}
				<div className={ 'sui-flushed wizard-settings__section ' + ( ! zeroScoreLayout ? 'sui-box-settings-row' : '' ) }>
					<div className="wizard-settings__section-content">
						<div className="wizard-settings__section-heading">
							<h2>{ __( 'Applied Config', 'wphb' ) }</h2>
						</div>
						<div className="sui-table-wrapper">
							<div className="sui-table-header sui-table-row">
								<div className="sui-table-header-item sui-table-col">{ __( 'Modules', 'wphb' ) }</div>
								<div className="sui-table-header-item sui-table-col">{ __( 'Settings applied', 'wphb' ) }</div>
								<div className="sui-table-header-item sui-table-col">{ __( 'Status', 'wphb' ) }</div>
							</div>
							<div className="sui-table-body">
								<div className="sui-table-row">
									<div className="sui-table-col sui-table-item-title">{ __( 'Asset Optimization', 'wphb' ) }</div>
									<div className="sui-table-col">
										{ ! this.props.isNetworkAdmin && <div>{ __( 'Compress files & clean code', 'wphb' ) }</div> }
										{ ! this.props.isNetworkAdmin && <div>{ __( 'Combine JavaScript & CSS', 'wphb' ) }</div> }
										{ ! this.props.isNetworkAdmin && <div>{ __( 'Swap Web Fonts', 'wphb' ) }</div> }
										{ ! this.props.isNetworkAdmin && <div>{ __( 'Delay JS Executions', 'wphb' ) } { ! this.props.isMember && <Tag type="purple sui-tag-sm" value={ __( 'Pro', 'wphb' ) } /> }</div> }
										{ ! this.props.isNetworkAdmin && <div>{ __( 'Generate Critical CSS', 'wphb' ) } { ! this.props.isMember && <Tag type="purple sui-tag-sm" value={ __( 'Pro', 'wphb' ) } /> }</div> }
										{ this.props.isNetworkAdmin && <div>{ __( 'Active on subsites', 'wphb' ) }</div> }
									</div>
									<div className="sui-table-col">
										{ ! this.props.isNetworkAdmin && this.renderStatusTag( this.props.settings.aoCompress ) }
										{ ! this.props.isNetworkAdmin && this.renderStatusTag( this.props.settings.aoCombine ) }
										{ ! this.props.isNetworkAdmin && this.renderStatusTag( this.props.settings.fontSwap ) }
										{ ! this.props.isNetworkAdmin && this.renderStatusTag( this.props.settings.delayJS, true ) }
										{ ! this.props.isNetworkAdmin && this.renderStatusTag( this.props.settings.criticalCSS, true ) }
										{ this.props.isNetworkAdmin && this.renderStatusTag( this.props.settings.aoEnable ) }
									</div>
								</div>
								<div className="sui-table-row">
									<div className="sui-table-col sui-table-item-title">{ __( 'CDN', 'wphb' ) }</div>
									<div className="sui-table-col">
										<div>{ __( 'CDN Network', 'wphb' ) } { ! this.props.isMember && <Tag type="purple sui-tag-sm" value={ __( 'Pro', 'wphb' ) } /> }</div>
									</div>
									<div className="sui-table-col">
										{ this.props.settings.aoCdn && this.props.isMember &&
											<Tag type="blue sui-tag-sm" value={ __( 'On', 'wphb' ) } /> }
										{ ! this.props.settings.aoCdn &&
											<Tag type="grey sui-tag-sm" value={ __( 'Disabled', 'wphb' ) } /> }
									</div>
								</div>

								{ this.props.settings.cacheEnable &&
									<div className="sui-table-row">
										<div className="sui-table-col sui-table-item-title">{ __( 'Page Caching', 'wphb' ) }</div>
										<div className="sui-table-col">
											<div>{ __( 'Preload caching on Homepage', 'wphb' ) }</div>
											{ ! this.props.settings.fastCGI && <div>{ __( 'Cache on mobile devices', 'wphb' ) }</div> }
											<div>{ __( 'Clear cache after new comment', 'wphb' ) }</div>
											<div>{ __( 'Show clear cache button in admin bar', 'wphb' ) }</div>
										</div>
										<div className="sui-table-col">
											{ this.renderStatusTag( this.props.settings.preloadHomepage ) }
											{ ! this.props.settings.fastCGI && this.renderStatusTag( this.props.settings.cacheOnMobile ) }
											{ this.renderStatusTag( this.props.settings.clearOnComment ) }
											{ this.renderStatusTag( this.props.settings.clearCacheButton ) }
										</div>
									</div>
								}

								<div className="sui-table-row">
									<div className="sui-table-col sui-table-item-title">{ __( 'Advanced Tools', 'wphb' ) }</div>
									<div className="sui-table-col">
										<div>{ __( 'Remove query strings from assets', 'wphb' ) }</div>
										<div>{ __( 'Remove default Emoji JS & CSS files', 'wphb' ) }</div>
										<div>{ __( 'Lazy load comments', 'wphb' ) }</div>
										<div>{ __( 'Optimize viewport meta tag', 'wphb' ) }</div>
										{ this.props.hasWoo &&
											<div>{ __( 'Faster non-shop pages', 'wphb' ) }</div>
										}
									</div>
									<div className="sui-table-col">
										{ this.renderStatusTag( this.props.settings.queryStrings ) }
										{ this.renderStatusTag( this.props.settings.removeEmoji ) }
										{ this.renderStatusTag( this.props.settings.lazyLoadComments ) }
										{ this.renderStatusTag( this.props.settings.viewPortMeta ) }
										{ this.props.hasWoo &&
											<React.Fragment>
												{ this.renderStatusTag( this.props.settings.cartFragments ) }
											</React.Fragment> }
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
				{ this.props.settings.desktopScore <= 0 && this.props.settings.mobileScore <= 0 &&
				( ! this.props.isMember && ( <UpsellCard rightTag={ true } onTrack={ this.props.trackUpsell } /> ) )
				}
			</Fragment>
		);
	}

	/**
	 * Get content.
	 *
	 * @return {JSX.Element} Content
	 */
	getContent() {
		let description;

		if ( 1 === this.props.step ) {
			description = __( 'Activate recommended features now, or skip and customize later.', 'wphb' );
			if ( this.props.showConflicts ) {
				description = __( 'Any issue reported here may cause issues while we set up the plugin.', 'wphb' );
				if ( ! this.props.issues.advCacheFile ) {
					description = __( 'There are no more potential issues. You can proceed with the setup.', 'wphb' );
				}
			}
		} else if ( 2 === this.props.step ) {
			description = __( "Hummingbird's Asset Optimization engine can combine and minify the files your website outputs when a user visits your website. The fewer requests your visitors have to make to your server, the better.", 'wphb' );
		} else if ( 3 === this.props.step ) {
			description = __( "Uptime monitors your server response time and lets you know when your website is down or too slow for your visitors. Monitor your site every minute to make sure it's up and graph your site speed so you can make sure everything is running super smooth.", 'wphb' );
		} else if ( 4 === this.props.step ) {
			description = __( 'Schedule performance reports and downtime alerts, including scan frequency, alert recipients, and downtime threshold settings.', 'wphb' );
		}

		return (
			<React.Fragment>
				{ 1 === this.props.step &&
					<div className="wizard-settings__intro">
						<span className="sui-icon-wand-magic" aria-hidden="true"></span>
						<p className={
							classNames(
								'sui-description',
								{ 'wphb-step-1-description': this.props.step === 1 }
							)
						}>
							{ description }
						</p>
					</div>

				}

				{ 1 === this.props.step && ! this.props.showConflicts &&
					<div className="sui-border-frame wphb-getting-started-setup">
						<Toggle
							id="tracking"
							onChange={ this.props.updateSettings }
							text={
								__( 'Help us make Hummingbird better', 'wphb' )
							}
							checked={ this.props.settings.tracking }
							tag={
								__( 'Recommended', 'wphb' )
							}
							description={
								<>
									{ __( 'Join 18,000+ users helping us improve with anonymous data. No personal info collected.', 'wphb' ) }
									{ ! this.props.isWhiteLabeled && (
										<>
											{ ' ' }
											<a href={ getLink( 'tracking' ) } target="_blank" rel="noreferrer">
												{ __( 'Learn more', 'wphb' ) }
											</a>
										</>
									) }
								</>
							}
						/>
					</div> }

				{ 1 === this.props.step && this.props.showConflicts &&
					<div className="wphb-progress-wrapper">
						{ this.getCompatPluginsContent() }
					</div> }

				{ 2 === this.props.step && this.getCustomizeSettings() }
				{ 3 === this.props.step && this.showResults() }
				{ 4 === this.props.step && this.notificationsSettings() }
			</React.Fragment>
		);
	}

	/**
	 * Handle "Continue" button click.
	 */
	continueToNextStep() {
		if ( 2 === this.props.step ) {
			this.props.nextStep().then( () => {
				this.setState( { scanning: true } );
			} );
		} else {
			this.props.nextStep();
		}
	}

	/**
	 * Render status tag based on condition and membership requirement.
	 *
	 * @param {boolean} condition The condition to check
	 * @param {boolean} isMember  Whether the feature requires membership
	 * @return {JSX.Element}      Tag component
	 */
	renderStatusTag( condition, isMember = false ) {
		const isEnabled = condition && ( ! isMember || this.props.isMember );
		return (
			<Tag
				type={ isEnabled ? 'blue sui-tag-sm' : 'grey sui-tag-sm' }
				value={ isEnabled ? __( 'On', 'wphb' ) : __( 'Disabled', 'wphb' ) }
			/>
		);
	}

	/**
	 * Render component.
	 *
	 * @return {JSX.Element}  Requirements component.
	 */
	render() {
		return (
			<Fragment>
				<div className="sui-row-with-sidenav ">
					<Box
						boxClass={ classNames( { 'box-setup-requirements': 1 === this.props.step }, 'content-step-' + this.props.step ) }
						hideHeader={ true }
						loading={ this.props.loading }
						loadingText={ 1 === this.props.step ? __( 'Checking status', 'wphb' ) : __( 'Saving settings', 'wphb' ) }
						content={ <React.Fragment>
							{ this.getContent() }
						</React.Fragment> } />
					{ this.getNavigation() }
				</div>
				<ProcessingConfigurationModal { ...this.props } cancelScanning={ this.cancelScanning.bind( this ) } />
			</Fragment>
		);
	}
}
