/**
 * External dependencies
 */
import React from 'react';

/**
 * Simple Radio Image Option Component
 *
 * @param {Object}   props             - Component props
 * @param {string}   props.id          - The unique ID for the radio input
 * @param {string}   props.name        - The name attribute for the radio group
 * @param {string}   props.value       - The value of this radio option
 * @param {string}   props.label       - The label text for the radio option
 * @param {string}   props.imageSrc    - The source URL for the image
 * @param {string}   props.imageSrcSet - The srcset for responsive images
 * @param {string}   props.imageAlt    - Alt text for the image
 * @param {string}   props.description - Description text below the option
 * @param {boolean}  props.checked     - Whether this option is selected
 * @param {Function} props.onChange    - Callback when selection changes
 *
 * @return {JSX.Element} Radio image option component
 */
const RadioImageOption = ( {
	id,
	name,
	value,
	label,
	imageSrc,
	imageSrcSet,
	imageAlt,
	description,
	checked = false,
	onChange
} ) => {
	return (
		<label
			htmlFor={ id }
			className={ `sui-radio-image wizard-settings__radio${ checked ? ' sui-radio-checked' : '' }` }
		>
			<span className="sui-radio">
				<input
					type="radio"
					name={ name }
					id={ id }
					value={ value }
					checked={ checked }
					onChange={ onChange }
					aria-labelledby={ `${ id }-label` }
				/>
				<span aria-hidden="true"></span>
				<span id={ `${ id }-label` } className="sui-radio-label">{ label }</span>
			</span>
			<img
				className="sui-radio-image__img"
				alt={ imageAlt }
				src={ imageSrc }
				srcSet={ imageSrcSet } />
			{ description && (
				<span className="sui-description">{ description }</span>
			) }
		</label>
	);
};

export default RadioImageOption;
