/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import React from 'react';

/**
 * Processing Configuration Modal Component
 *
 * A React functional component that displays a modal during the setup
 * configuration process, showing progress updates and module statuses.
 *
 * @param {Object} props - All component properties passed from wizard
 *
 * @return {JSX.Element} The modal JSX element
 */
export default function ProcessingConfigurationModal( props ) {
	const progress = 0; // Static for now

	/**
	 * Handle cancel configuration action
	 */
	const cancelConfiguration = () => {
		// Access the cancelScanning method from props if available
		if ( props.cancelScanning ) {
			props.cancelScanning();
		}
	};
	return (
		<div className="sui-modal sui-modal-lg">
			<div
				role="dialog"
				id="wphb-processing-configuration-modal"
				className="sui-modal-content"
				aria-modal="true"
				aria-labelledby="runProcessingConfiguration"
				aria-describedby="dialogDescription"
			>
				<div className="sui-box">
					<div className="sui-box-header">
						<h3 className="sui-box-title" id="runProcessingConfiguration">
							{ __( 'Processing Configuration', 'wphb' ) }
						</h3>
					</div>

					<div className="sui-box-body wphb-performance-scan-modal">
						<p id="dialogDescription">
							{ __( 'Applying settings and running a performance test to reveal your site\'s new speed.', 'wphb' ) }
						</p>

						<div className="sui-progress-block">
							<div className="sui-progress">
								<span className="sui-progress-icon" aria-hidden="true">
									<span className="sui-icon-loader sui-loading"></span>
								</span>
								<div className="sui-progress-text">
									<span>{ progress }%</span>
								</div>
								<div className="sui-progress-bar" aria-hidden="true">
									<span style={ { width: `${ progress }%` } }></span>
								</div>
							</div>
							<button
								className="sui-button-icon sui-tooltip"
								id="cancel-configuration-check"
								onClick={ cancelConfiguration }
								type="button"
								data-tooltip={ __( 'Cancel Processing', 'wphb' ) }
							>
								<span className="sui-icon-close" aria-hidden="true"></span>
							</button>
						</div>

						<div className="sui-progress-state">
							<span className="sui-progress-state-text">
								{ __( 'Preparing...', 'wphb' ) }
							</span>
							<div className="module-status-wrap">
								<ul className="module-status gap-10px">
									{ ! props.isNetworkAdmin && (
										<li className="scanning_assets font-medium mb-0">
											<i className="grey-circle"></i>
											{ __( 'Scanning Assets', 'wphb' ) }
										</li>
									) }
									{ ( props.settings.criticalCSS && ! props.isNetworkAdmin ) && (
										<li className={ `generating_critical_css font-medium mb-0 ${ ! props.isMember ? 'disabled' : '' }` }>
											<i className="grey-circle"></i>
											{ __( 'Generating Critical CSS', 'wphb' ) }
											{ ! props.isMember && (
												<span className="sui-tag sui-tag-pro">
													{ __( 'Pro', 'wphb' ) }
												</span>
											) }
										</li>
									) }
									{ props.settings.cacheEnable && (
										<li className="generating_page_cache font-medium mb-0">
											<i className="grey-circle"></i>
											{ __( 'Generating Page cache for homepage', 'wphb' ) }
										</li>
									) }
									<li className="running_performance_test font-medium mb-0">
										<i className="grey-circle"></i>
										{ __( 'Running Performance Test', 'wphb' ) }
									</li>
								</ul>
							</div>
							<button
								onClick={ cancelConfiguration }
								className="sui-button sui-button-ghost"
								style={ { width: '170px', marginTop: '20px' } }
							>
								{ __( 'Cancel Configuration', 'wphb' ) }
							</button>
						</div>
					</div>
				</div>
			</div>
		</div>
	);
}
