import React, { useState } from 'react';

/**
 * WordPress dependencies
 */
const { __ } = wp.i18n;

/**
 * Internal dependencies
 */
import Tabs from '../../components/sui-tabs';
import Button from '../../components/sui-button';
import { getLink } from '../../../js/utils/helpers';
import CircularProgress from './CircularProgress';
import UpsellCard from './UpsellCard';

/**
 * PerformanceCard Component
 *
 * Displays performance score and pro upgrade options using SUI framework classes
 *
 * @param {Object}   props              - Component props
 * @param {number}   props.desktopScore - Desktop performance score
 * @param {number}   props.mobileScore  - Mobile performance score
 * @param {boolean}  props.isMember     - Whether to show pro upgrade section
 * @param {string}   props.lastScan     - Last scan time text
 * @param {Function} props.onTrack      - Function to call when "Unlock Pro" button is clicked
 * @return {JSX.Element} Performance card with score display and pro upgrade section
 */
export default function PerformanceCard( {
	desktopScore = 0,
	mobileScore = 0,
	isMember = false,
	lastScan = __( 'a minute ago', 'wphb' ),
	onTrack = () => {},
} = {} ) {
	const [ activeTab, setActiveTab ] = useState( 'desktop' );
	const score = activeTab === 'desktop' ? desktopScore : mobileScore; // Use props for dynamic score

	const handleTabClick = ( tabName ) => {
		setActiveTab( tabName );
	};

	const renderPerformanceScore = () => (
		<div className="wizard-performance-score">
			<div>
				<p className="sui-box-title">
					{ __( 'Performance', 'wphb' ) }<br />{ __( 'Score', 'wphb' ) }
				</p>
			</div>
			<div className="sui-circle-score-wrapper">
				<CircularProgress score={ score } />
				<div className="wizard-performance-score">
					{ score }
				</div>
			</div>
			<div>
				<span className="sui-description">
					{ __( 'Last Scan', 'wphb' ) }<br />{ lastScan }
				</span>
			</div>
		</div>
	);

	const sideTabs = [
		{
			title: __( 'Desktop', 'wphb' ),
			checked: activeTab === 'desktop',
			id: 'desktop',
			onClick: () => handleTabClick( 'desktop' )
		},
		{
			title: __( 'Mobile', 'wphb' ),
			checked: activeTab === 'mobile',
			id: 'mobile',
			onClick: () => handleTabClick( 'mobile' )
		}
	];

	return (
		<div className="wizard-performance-card">
			<div className="sui-box-score">
				<Tabs sideTabs="true" menu={ sideTabs } />
				<div className="sui-tab-content">
					{ renderPerformanceScore() }
				</div>
			</div>

			{ ! isMember && ( <UpsellCard onTrack={ onTrack } /> ) }
		</div>
	);
}

