import React, { useEffect, useRef } from 'react';

/**
 * External dependencies
 */
import '@wpmudev/shared-ui/js/scores';

/**
 * CircularProgress Component
 *
 * A circular progress component that displays score with SUI styling and tooltip support
 *
 * @param {Object}  props              - Component props
 * @param {number}  props.score        - Progress score (0-100)
 * @param {boolean} props.large        - Whether to use large size variant (default: true)
 * @param {number}  props.tooltipWidth - Custom tooltip width in pixels (default: 280)
 *
 * @return {JSX.Element} Circular progress component
 */
export default function CircularProgress( {
	score = 0,
	large = true,
	tooltipWidth = 280
} ) {
	/**
	 * Get impact score class based on score
	 *
	 * @param {number} scoreValue - Performance score (0-100)
	 * @return {string} Impact score class
	 */
	const getImpactScoreClass = ( scoreValue ) => {
		if ( scoreValue >= 90 ) {
			return 'success';
		} else if ( scoreValue >= 50 ) {
			return 'warning';
		}

		return 'error';
	};

	const circleRef = useRef( null );

	// Auto-generate impact score class if not provided
	const finalImpactScoreClass = getImpactScoreClass( score );

	const sizeClass = large ? 'sui-circle-score-lg' : '';
	const gradeClass = finalImpactScoreClass ? `sui-grade-${ finalImpactScoreClass }` : '';

	const style = {
		'--tooltip-width': `${ tooltipWidth }px`
	};

	// Initialize circle score after component mounts and when score changes
	useEffect( () => {
		if ( circleRef.current && window.SUI && window.SUI.loadCircleScore ) {
			// Clear any existing content and SUI data attributes to prevent stacking
			circleRef.current.innerHTML = '';

			// Remove any SUI-related data attributes that might cause caching issues
			circleRef.current.removeAttribute( 'data-score' );

			// Set the data attributes after clearing
			circleRef.current.setAttribute( 'data-score', parseInt( score ) );

			if ( circleRef.current ) {
				window.SUI.loadCircleScore( circleRef.current );
			}
		}
	}, [ score ] );

	return (
		<div
			ref={ circleRef }
			key={ `circle-${ score }` }
			style={ style }
			className={ `sui-circle-score ${ sizeClass } ${ gradeClass }`.trim() }
		>
		</div>
	);
}
