/**
 * External dependencies
 */
import React from 'react';
import ReactDOM from 'react-dom';
import PropTypes from 'prop-types';

/**
 * WordPress dependencies
 */
import domReady from '@wordpress/dom-ready';
const { __ } = wp.i18n;

/**
 * Internal dependencies
 */
require( '../../js/mixpanel' );
import '../app.scss';
import { getLink } from '../../js/utils/helpers';
import HBAPIFetch from '../api';
import Button from '../components/sui-button';
import ButtonLoading from '../components/sui-button-loading';
import Tooltip from '../components/sui-tooltip';
import Wizard from '../views/setup/wizard';
import HBFetcher from '../../js/utils/fetcher';

/**
 * SetupWizard component.
 *
 * @since 3.3.1
 */
class SetupWizard extends React.Component {
	/**
	 * Component constructor.
	 *
	 * @param {Object} props
	 */
	constructor( props ) {
		super( props );

		this.state = {
			api: new HBAPIFetch(),
			isMember: this.props.wphbData.isMember,
			hasUptime: this.props.wphbData.hasUptime,
			loading: false,
			checkDocumentation: false,
			checkUpsell: false,
			/**
			 * Steps:
			 * 1. Start of setup
			 * 2. Asset optimization
			 * 3. Uptime
			 * 4. Page caching
			 * 5. Advanced tools
			 * 6. Finish
			 */
			step: 1,
			issues: {
				advCacheFile: false
			},
			showConflicts: false,
			settings: {
				aoEnable: true,
				aoCompress: true,
				aoCombine: true,
				aoCdn: Boolean( this.props.wphbData.isMember ),
				delayJS: false,
				criticalCSS: false,
				fontSwap: true,
				uptimeEnable: Boolean( this.props.wphbData.hasUptime ),
				cacheEnable: true,
				fastCGI: Boolean( this.props.wphbData.isFastCGISupported ) ? true : false,
				isFastCGISupported: Boolean( this.props.wphbData.isFastCGISupported ),
				cacheOnMobile: true,
				clearOnComment: true,
				cacheHeader: true,
				clearCacheButton: true,
				queryStrings: true,
				cartFragments: Boolean( this.props.wphbData.hasWoo ),
				removeEmoji: true,
				tracking: false,
				notifications: {
					report: {
						enabled: Boolean( this.props.wphbData.isConnected ) ? true : false,
						frequency: 'daily',
						time: '12:00',
						weekDay: 'Monday',
						monthDay: '1',
						recipients: [],
					},
					alert: {
						enabled: Boolean( this.props.wphbData.isConnected ) ? true : false,
						downtime: Boolean( this.props.wphbData.isMember ) ? 'instant' : '5 mins',
						sameAsReportRecipient: true,
						recipients: [],
					}
				},
				preloadHomepage: true,
				lazyLoadComments: true,
				viewPortMeta: true,
				desktopScore: null,
				mobileScore: null,
			},
		};

		this.checkRequirements = this.checkRequirements.bind( this );
		this.removeAdvancedCache = this.removeAdvancedCache.bind( this );
		this.disableFastCGI = this.disableFastCGI.bind( this );
		this.skipConflicts = this.skipConflicts.bind( this );
		this.nextStep = this.nextStep.bind( this );
		this.prevStep = this.prevStep.bind( this );
		this.finish = this.finish.bind( this );
		this.updateSettings = this.updateSettings.bind( this );
		this.toggleModule = this.toggleModule.bind( this );
		this.quitWizard = this.quitWizard.bind( this );
		this.trackUpsell = this.trackUpsell.bind( this );
	}

	/**
	 * Wizard started.
	 */
	componentDidMount() {
		this.checkRequirements();
	}

	/**
	 * Skip conflict check.
	 */
	skipConflicts() {
		this.setState( {
			showConflicts: false,
			step: 2
		} );
	}

	/**
	 * Go to next step in wizard.
	 *
	 * @param {boolean} ignoreSteptwo Ignore step two (AO settings) if already on step two.
	 */
	nextStep( ignoreSteptwo = true ) {
		if ( 1 === this.state.step && this.state.issues.advCacheFile ) {
			this.setState( { showConflicts: true } );
			return;
		}

		let step;
		if ( 2 === this.state.step && ignoreSteptwo ) {
			step = this.state.step;
		} else {
			step = this.state.step + 1;
		}

		if ( ! ignoreSteptwo ) {
			this.setState( { step } );
			return;
		}

		this.setState( { loading: true } );

		const data = { ...this.state.settings, module: '', enable: false };
		if ( 2 === this.state.step ) {
			data.module = 'customize_features';
			data.enable = this.state.settings.aoEnable;
		}

		return this.state.api
			.post( 'settings', data )
			.then( () => this.setState( {
				showConflicts: false,
				step,
				loading: false
			} ) )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Get performance score.
	 *
	 * @param {boolean} ignoreSteptwo Ignore step two (AO settings) if already on step two.
	 */
	getPerformanceScore( ignoreSteptwo ) {
		const settings = { ...this.state.settings };

		this.state.api
			.post( 'get_performance_score' )
			.then( ( response ) => {
				settings.desktopScore = response.desktop;
				settings.mobileScore = response.mobile;
				this.setState( {
					settings,
				} );
				this.nextStep( ignoreSteptwo );
			} )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Go to previous step in wizard.
	 */
	prevStep() {
		const step = this.state.step - 1;

		this.setState( { step } );
	}

	// ...existing code...

	saveNotificationsSettings() {
		const { settings } = this.state;

		const getRecipients = ( type ) => {
			if (
				settings.notifications &&
        settings.notifications[ type ] && settings.notifications[ type ].enabled
			) {
				if (
					'alert' === type &&
            true === settings.notifications.alert.sameAsReportRecipient
				) {
					return settings.notifications.report?.recipients || [];
				}
				if ( Array.isArray( settings.notifications[ type ].recipients ) ) {
					return settings.notifications[ type ].recipients;
				}
			}
			return [];
		};

		// PERFORMANCE REPORTS
		const performanceReportsData = {
			view: 'schedule',
			module: 'performance',
			type: 'reports',
			schedule: {
				frequency: settings.notifications?.report?.frequency || 7,
				time: settings.notifications?.report?.time || '6:00',
				weekDay: settings.notifications?.report?.weekDay || 'Tuesday',
				monthDay: settings.notifications?.report?.monthDay || '11',
				threshold: '',
			},
			recipients: getRecipients( 'report' ),
			performance: {
				device: 'both',
				metrics: true,
				audits: true,
				fieldData: true,
			},
			uptime: {
				showPing: true,
			},
			database: {
				revisions: true,
				drafts: true,
				trash: true,
				spam: true,
				trashComment: true,
				expiredTransients: true,
				transients: false,
			},
		};

		// UPTIME REPORTS
		const uptimeReportsData = {
			view: 'schedule',
			module: 'uptime',
			type: 'reports',
			schedule: {
				frequency: settings.notifications?.report?.frequency || 1,
				time: settings.notifications?.report?.time || '3:00',
				weekDay: settings.notifications?.report?.weekDay || 'Tuesday',
				monthDay: settings.notifications?.report?.monthDay || '5',
				threshold: '',
			},
			recipients: getRecipients( 'report' ),
			performance: {
				device: 'both',
				metrics: true,
				audits: true,
				fieldData: true,
			},
			uptime: {
				showPing: true,
			},
			database: {
				revisions: true,
				drafts: true,
				trash: true,
				spam: true,
				trashComment: true,
				expiredTransients: true,
				transients: false,
			},
		};

		// UPTIME NOTIFICATIONS
		const rawThreshold = settings.notifications?.alert?.downtime ?? 'instant';
		let threshold;
		if ( rawThreshold === 'instant' ) {
			threshold = 0;
		} else {
			threshold = parseInt( rawThreshold, 10 ) || 5;
		}
		const uptimeNotificationsData = {
			view: 'schedule',
			module: 'uptime',
			type: 'notifications',
			schedule: {
				frequency: settings.notifications?.alert?.frequency || 7,
				time: settings.notifications?.alert?.time || false,
				threshold,
				weekDay: settings.notifications?.alert?.weekDay || false,
				monthDay: settings.notifications?.alert?.monthDay || false,
			},
			recipients: getRecipients( 'alert' ),
			performance: {
				device: 'both',
				metrics: true,
				audits: true,
				fieldData: true,
			},
			uptime: {
				showPing: true,
			},
			database: {
				revisions: true,
				drafts: true,
				trash: true,
				spam: true,
				trashComment: true,
				expiredTransients: true,
				transients: false,
			},
		};

		// DATABASE REPORTS
		const databaseReportsData = {
			view: 'schedule',
			module: 'database',
			type: 'reports',
			schedule: {
				frequency: settings.notifications?.report?.frequency || 1,
				time: settings.notifications?.report?.time || '19:00',
				weekDay: settings.notifications?.report?.weekDay || 'Friday',
				monthDay: settings.notifications?.report?.monthDay || '27',
				threshold: '',
			},
			recipients: getRecipients( 'report' ),
			performance: {
				device: 'both',
				metrics: true,
				audits: true,
				fieldData: true,
			},
			uptime: {
				showPing: true,
			},
			database: {
				revisions: true,
				drafts: true,
				trash: true,
				spam: true,
				trashComment: true,
				expiredTransients: true,
				transients: true,
			},
		};

		const notificationsDataArray = [
			databaseReportsData,
			performanceReportsData,
			uptimeReportsData,
			uptimeNotificationsData,
		];

		const sendNotificationRequests = async () => {
			for ( const dataObj of notificationsDataArray ) {
				// Normalize frequency
				if ( dataObj.schedule && typeof dataObj.schedule.frequency === 'string' ) {
					switch ( dataObj.schedule.frequency.toLowerCase() ) {
						case 'weekly':
							dataObj.schedule.frequency = 7;
							break;
						case 'monthly':
							dataObj.schedule.frequency = 30;
							break;
						default:
							dataObj.schedule.frequency = 1;
							break;
					}
				}

				try {
					await HBFetcher.notifications.enable( dataObj, true );
					// console.log(response);
				} catch ( error ) {
					window.console.log( error );
				}
				// 0.1 seconds break after each ajax call.
				await new Promise( ( resolve ) => setTimeout( resolve, 100 ) );
			}
		};

		return sendNotificationRequests();
	}

	/**
	 * Complete wizard.
	 *
	 * @param {string} goToPage Go to page.
	 */
	async finish( goToPage = 'pluginDash' ) {
		this.setState( { loading: true } );
		if ( 'string' !== typeof goToPage ) {
			goToPage = 'pluginDash';
		}

		this.trackSetupEvents( goToPage );
		await this.saveNotificationsSettings(); // Wait for notifications to finish
		this.state.api
			.post( 'complete_wizard', goToPage )
			.then( () => {
				if ( 'configs' === goToPage ) {
					goToPage = 'runPerf';
				}
				window.location.href = getLink( goToPage );
			} )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Check setup wizard requirements.
	 *
	 * @param {boolean} setLoadingState
	 */
	checkRequirements( setLoadingState = false ) {
		if ( setLoadingState ) {
			this.setState( { loading: true } );
		}

		this.state.api
			.post( 'check_requirements' )
			.then( ( response ) => {
				this.setState( {
					loading: false,
					issues: response.status
				} );
			} )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Remove advanced-cache.php file.
	 */
	removeAdvancedCache() {
		this.setState( { loading: true } );

		this.state.api
			.post( 'remove_advanced_cache' )
			.then( ( response ) => {
				this.setState( {
					loading: false,
					issues: response.status
				} );
			} )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Disable FastCGI cache.
	 */
	disableFastCGI() {
		this.setState( { loading: true } );

		this.state.api
			.post( 'disable_fast_cgi' )
			.then( ( response ) => {
				this.setState( {
					loading: false,
					issues: response.status
				} );
			} )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Update settings on toggle status change.
	 *
	 * @param {Object} e
	 */
	updateSettings = ( e ) => {
		this.setState( ( prevState ) => {
			const settings = { ...prevState.settings };
			const settingKey = e.target.type === 'radio' ? e.target.name : e.target.id;
			let value;

			if ( typeof e.target.checked !== 'undefined' && e.target.type !== 'radio' ) {
				value = e.target.checked;
			} else if ( typeof e.target.value !== 'undefined' ) {
				value = e.target.value;

				// Special handling for CDN radio button - convert to boolean
				if ( e.target.name === 'aoCdn' && e.target.type === 'radio' ) {
					value = e.target.value === 'global';
				}
			} else {
				value = e.target.value;
			}

			if ( settingKey === 'notifications' ) {
				// Merge deeply into notifications
				settings.notifications = {
					...settings.notifications,
					...value,
					report: {
						...settings.notifications.report,
						...( value.report || {} )
					},
					alert: {
						...settings.notifications.alert,
						...( value.alert || {} )
					}
				};
			} else {
				settings[ settingKey ] = value;
			}

			if ( e.target.id === 'tracking' ) {
				this.trackUserConsentToggle( e.target.checked );
			}

			return { settings };
		} );
	};

	/**
	 * Process enable/disable button clicks.
	 *
	 * @param {string} action  Action: enable|disable.
	 * @param {string} setting Setting ID.
	 */
	toggleModule( action, setting ) {
		const settings = { ...this.state.settings };
		settings[ setting ] = 'enable' === action;

		if ( 'cacheEnable' === setting && ( 'enable' === action || 'disable' === action ) ) {
			settings.fastCGI = false;
		}

		this.setState( { settings } );
	}

	/**
	 * Quit wizard.
	 * TODO: add tracking
	 */
	quitWizard() {
		this.setState( { loading: true } );
		this.trackSetupEvents();
		this.state.api.post( 'cancel_wizard' )
			.then( () => {
				window.location.href = getLink( 'pluginDash' );
			} )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Track setup wizard events.
	 *
	 * @param {string} action
	 */
	trackSetupEvents( action = 'quit' ) {
		const actionMap = {
			configs: 'apply_configs',
			runPerf: 'performance_test',
			pluginDash: 'complete_wizard',
			hubConnect: 'connect',
		};
		action = actionMap[ action ] || action;

		const stepMap = {
			1: this.state.issues.advCacheFile ? 'conflict' : 'tracking',
			2: 'customize',
			3: 'summary',
			4: 'reports_alerts',
		};
		const quitStep = 'quit' === action ? stepMap[ this.state.step ] || 'na' : 'na';

		const aoSettings = {
			aoCombine: 'combine',
			aoCompress: 'compress',
			aoCdn: 'cdn',
			delayJS: 'js_delay',
			criticalCSS: 'critical_css',
			fontSwap: 'font_swap',
		};
		const enabledAoFeatures = this.state.settings.aoEnable ? this.mapSettings( aoSettings ) : 'disabled';

		const cacheSettings = {
			preloadHomepage: 'preload_homepage',
			clearOnComment: 'clear_cache_comment',
			clearCacheButton: 'show_cache_button',
		};
		const enabledCacheFeatures = this.state.settings.cacheEnable ? this.mapSettings( cacheSettings ) : 'disabled';
		if ( this.state.settings.cacheEnable ) {
			if ( this.state.settings.fastCGI ) {
				enabledCacheFeatures.push( 'static_server_cache' );
			} else {
				enabledCacheFeatures.push( 'local_page_cache' );
				enabledCacheFeatures.push( 'mobile_cache' );
			}
		}

		const advancedSettings = {
			queryStrings: 'remove_query_strings',
			cartFragments: 'disable_cart_fragments',
			removeEmoji: 'remove_emoji',
			lazyLoadComments: 'lazy_comment',
			viewPortMeta: 'viewport_meta',
		};
		const advancedFeaturesStatus = this.mapSettings( advancedSettings );

		const enabledAdvancedFeatures = advancedFeaturesStatus.length > 0 ? advancedFeaturesStatus : 'disabled';

		if ( this.state.settings.tracking ) {
			window.wphbMixPanel.optIn();
		}

		const enabledOptionsArr = []
			.concat(
				Array.isArray( enabledAoFeatures ) ? enabledAoFeatures : [],
				Array.isArray( enabledCacheFeatures ) ? enabledCacheFeatures : [],
				Array.isArray( enabledAdvancedFeatures ) ? enabledAdvancedFeatures : []
			)
			.filter( Boolean );

		const enabledOptions = enabledOptionsArr.length > 0 ? enabledOptionsArr : 'na';

		let enabledNotifications = [];
		if ( this.state.settings.notifications ) {
			if ( this.state.settings.notifications.report && this.state.settings.notifications.report.enabled ) {
				enabledNotifications.push( 'scheduled_reports' );
			}
			if ( this.state.settings.notifications.alert && this.state.settings.notifications.alert.enabled ) {
				enabledNotifications.push( 'downtime_alert' );
			}
		}
		enabledNotifications = enabledNotifications.length > 0 ? enabledNotifications : 'na';

		let performanceScore = [];
		if ( this.state.settings.desktopScore ) {
			performanceScore.push( 'desktop_score' );
		}
		if ( this.state.settings.mobileScore ) {
			performanceScore.push( 'mobile_score' );
		}
		performanceScore = performanceScore.length > 0 ? performanceScore : 'na';

		window.wphbMixPanel.track( 'Setup Wizard New', {
			Action: action,
			'Quit Step Location': quitStep,
			'Enabled Options': this.state.step > 1 ? enabledOptions : 'na',
			Reports: this.state.step > 3 ? enabledNotifications : 'na',
			'Performance Score': this.state.step > 2 ? performanceScore : 'na',
			Upsell: this.state.checkUpsell ? 'clicked' : 'not_clicked', // todo
		} );

		if ( this.state.settings.cacheEnable ) {
			window.wphbMixPanel.trackPageCachingSettings(
				'activate',
				this.state.settings.fastCGI ? 'hosting_static_cache' : 'local_page_cache',
				'wizard',
				'na',
				this.state.settings.preloadHomepage ? 'enabled' : 'disabled',
			);
		}
	}

	/**
	 * Map settings to their respective names.
	 *
	 * @param {string} settingsMap
	 */
	mapSettings( settingsMap ) {
		return Object.keys( settingsMap ).filter( ( key ) => this.state.settings[ key ] ).map( ( key ) => settingsMap[ key ] );
	}

	trackDocumentation() {
		this.setState( { checkDocumentation: true } );
	}

	trackUpsell() {
		this.setState( { checkUpsell: true } );
	}

	/**
	 * Take action on user consent toggle.
	 *
	 * @param {string} tracking
	 */
	trackUserConsentToggle( tracking ) {
		this.state.api
			.post( 'track_user_consent_toggle', tracking )
			.then( () => {
				if ( tracking ) {
					window.wphbMixPanel.optIn();
				}
			} )
			.catch( ( error ) => window.console.log( error ) );
	}

	/**
	 * Get wizard header.
	 *
	 * @return {JSX.Element} Wizard header
	 */
	getHeader() {
		return (
			<div className="sui-header wphb-wizard-header">
				<h2 className="sui-header-title">
					<img
						className="sui-image"
						alt={ __( 'Setup wizard', 'wphb' ) }
						src={ getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/hummingbird.png' }
						srcSet={
							getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/hummingbird.png 1x, ' +
							getLink( 'wphbDirUrl' ) + 'admin/assets/image/setup/hummingbird@2x.png 2x'
						} />
					<div>
						{ __( 'Hummingbird', 'wphb' ) }
						<small>{ __( 'Wizard', 'wphb' ) }</small>
					</div>
				</h2>
				<div className="sui-actions-right">
					{ ! this.props.wphbData.isWhiteLabeled && (
						<Button
							classes={ [ 'sui-button-icon-right' ] }
							icon="sui-icon-open-new-window"
							target="blank"
							url={ getLink( 'docs' ) }
							onClick={ () => this.trackDocumentation() }
							text={ __( 'Documentation', 'wphb' ) } />
					) }
				</div>
			</div>
		);
	}

	/**
	 * Render component.
	 *
	 * @return {*} Gzip page.
	 */
	render() {
		return (
			<React.Fragment>
				{ this.getHeader() }
				<Wizard
					loading={ this.state.loading }
					step={ this.state.step }
					showConflicts={ this.state.showConflicts }
					issues={ this.state.issues }
					minifySteps={ this.props.wphbData.minifySteps }
					nextStep={ this.nextStep }
					getPerformanceScore={ this.getPerformanceScore.bind( this ) }
					prevStep={ this.prevStep }
					finish={ this.finish }
					quitWizard={ this.quitWizard }
					skipConflicts={ this.skipConflicts }
					isMember={ this.state.isMember }
					isNetworkAdmin={ this.props.wphbData.isNetworkAdmin }
					hasUptime={ this.state.hasUptime }
					settings={ this.state.settings }
					hasWoo={ this.props.wphbData.hasWoo }
					reCheckRequirements={ () => this.checkRequirements( true ) }
					updateSettings={ this.updateSettings }
					toggleModule={ this.toggleModule }
					disableFastCGI={ this.disableFastCGI }
					removeAdvancedCache={ this.removeAdvancedCache }
					isConnected={ this.props.wphbData.isConnected }
					isWhiteLabeled={ this.props.wphbData.isWhiteLabeled }
					trackUpsell={ this.trackUpsell }
				/>
			</React.Fragment>
		);
	}
}

SetupWizard.propTypes = {
	wphbData: PropTypes.object,
};

domReady( function() {
	const setupWizard = document.getElementById( 'wrap-wphb-setup' );
	if ( setupWizard ) {
		const setupReact = ReactDOM.render(
			/*** @var {object} window.wphb */
			<SetupWizard wphbData={ window.wphb } />,
			setupWizard
		);
	}
} );
