import React, { useState, useEffect } from 'react';
import { __ } from '@wordpress/i18n';

function AddRecipientsModal( { onClose, onAdd, existingRecipients = [] } ) {
	const [ activeTab, setActiveTab ] = useState( 'users' );
	const [ users, setUsers ] = useState( [] );
	const [ selected, setSelected ] = useState( [ ...existingRecipients ] );
	const [ name, setName ] = useState( '' );
	const [ email, setEmail ] = useState( '' );
	const [ emailError, setEmailError ] = useState( '' );
	const [ loading, setLoading ] = useState( false );

	useEffect( () => {
		initUserSelects();
		const fetchInitialUsers = async () => {
			setLoading( true );
			try {
				const response = await fetch( ajaxurl, {
					method: 'POST',
					headers: {
						'Content-Type': 'application/x-www-form-urlencoded',
					},
					body: new URLSearchParams( {
						action: 'wphb_pro_search_users',
						query: '',
						exclude: [],
						nonce: wphb.nonces.HBFetchNonce,
					} ),
				} );
				const result = await response.json();
				setUsers( result?.data || [] );
			} catch ( e ) {
				setUsers( [] );
			}
			setLoading( false );
		};
		fetchInitialUsers();
	}, [] );

	/**
	 * Initialize the search user select.
	 *
	 * @since 3.1.1
	 */
	const initUserSelects = () => {
		const userSelect = jQuery( '#search-users' );
		const exclude = Array.isArray( selected ) ? selected.map( ( u ) => u.email ) : [];

		userSelect.SUIselect2( {
			minimumInputLength: 3,
			maximumSelectionLength: 1,
			ajax: {
				url: typeof ajaxurl !== 'undefined' ? ajaxurl : window.ajaxurl,
				method: 'POST',
				dataType: 'json',
				delay: 250,
				data( params ) {
					return {
						action: 'wphb_pro_search_users',
						query: params.term,
						nonce: wphb.nonces.HBFetchNonce,
						exclude,
					};
				},
				processResults( data ) {
					return {
						results: jQuery.map(
							data.data || [],
							function( item, index ) {
								return {
									text: item.name,
									id: index,
									user: {
										name: item.name,
										email: item.email,
										role: item.role,
										avatar: item.avatar,
										id: item.id,
									},
								};
							}
						),
					};
				},
			},
		} );

		userSelect.on( 'select2:select', function( e ) {
			handleAddUser( e.params.data.user );
			userSelect.val( null ).trigger( 'change' );
		} );
	};

	const validateEmail = async ( email ) => {
		if ( ! email ) {
			return false;
		}
		try {
			const response = await fetch( ajaxurl, {
				method: 'POST',
				headers: {
					'Content-Type': 'application/x-www-form-urlencoded',
				},
				body: new URLSearchParams( {
					action: 'wphb_pro_get_avatar',
					email,
					nonce: wphb.nonces.HBFetchNonce,
				} ),
			} );
			const result = await response.json();

			return !! result.success;
		} catch ( e ) {
			return false;
		}
	};

	const handleRemoveUser = ( user ) => {
		const updated = selected.filter( ( u ) => u.email !== user.email );
		setSelected( updated );
		onAdd( updated );
	};

	const handleAddUser = ( user ) => {
		setSelected( ( prevSelected ) => {
			if ( ! prevSelected.some( ( u ) => u.email === user.email ) ) {
				const updated = [ ...prevSelected, user ];
				onAdd( updated );
				return updated;
			}
			return prevSelected;
		} );
	};

	const handleAddEmail = async () => {
		setEmailError( '' );
		if ( ! name || ! email ) {
			setEmailError( __( 'Name and email required.', 'wphb' ) );
			return;
		}
		setLoading( true );
		const valid = await validateEmail( email );
		setLoading( false );
		if ( ! valid ) {
			setEmailError( __( 'Invalid email address.', 'wphb' ) );
			return;
		}
		const avatar = window?.wphb?.links?.defaultAvatar || '';
		const newRecipient = { name, email, id: 0, role: '', avatar };
		handleAddUser( newRecipient );
		setName( '' );
		setEmail( '' );
	};

	const availableUsers = users.filter(
		( user ) => ! selected.some( ( selectedUser ) => selectedUser.email === user.email )
	);

	return (
		<div className="sui-modal sui-active sui-modal-lg">
			<div role="dialog" id="notification-modal" className="sui-modal-content sui-content-fade-in" aria-modal="true">
				<div id="notifications-slide-recipients" className="sui-box sui-modal-slide sui-active sui-loaded" data-modal-size="lg">
					<div className="sui-box-header sui-flatten sui-content-center sui-spacing-top--60">
						<button className="sui-button-icon sui-button-float--right" onClick={ onClose }>
							<span className="sui-icon-close sui-md" aria-hidden="true"></span>
							<span className="sui-screen-reader-text"> { __( 'Close this modal', 'wphb' ) } </span>
						</button>
						<h3 className="sui-box-title sui-lg"> { __( 'Recipients', 'wphb' ) } </h3>
						<p className="sui-description">
							{ __( 'Add as many recipients as you like. Each recipient will be notified of any downtime logs for the scheduled period.', 'wphb' ) }
						</p>
					</div>
					<div className="sui-box-body sui-spacing-top--20 sui-spacing-bottom--20">
						<div className="sui-tabs sui-tabs-overflow">
							<div role="tablist" className="sui-tabs-menu">
								<button type="button" role="tab" className={ `sui-tab-item${ activeTab === 'users' ? ' active' : '' }` } aria-selected={ activeTab === 'users' } onClick={ () => setActiveTab( 'users' ) }>
									{ __( 'Add users', 'wphb' ) }
								</button>
								<button type="button" role="tab" className={ `sui-tab-item${ activeTab === 'email' ? ' active' : '' }` } aria-selected={ activeTab === 'email' } onClick={ () => setActiveTab( 'email' ) }>
									{ __( 'Add by email', 'wphb' ) }
								</button>
							</div>
							<div className="sui-tabs-content">
								{ /* Add Users Tab */ }
								<div role="tabpanel" className={ `sui-tab-content${ activeTab === 'users' ? ' active' : '' }` } hidden={ activeTab !== 'users' }>
									<div className="sui-form-field sui-no-margin-bottom">
										<label htmlFor="search-users" className="sui-label"> { __( 'Search users', 'wphb' ) } </label>
										<select id="search-users" className="sui-select" data-theme="search" data-placeholder={ __( 'Type username', 'wphb' ) } multiple></select>
									</div>

									{ selected.length > 0 && (
										<div className="sui-margin-top">
											<strong>{ __( 'Added users', 'wphb' ) }</strong>
											<div className="sui-recipients" id="modal-user-recipients-list">
												{ selected.map( ( user ) => (
													<div className="sui-recipient" key={ user.email } data-id={ user.id } data-email={ user.email }>
														<span className="recipient-name">
															<span className="subscriber">
																<img src={ user.avatar } alt={ user.email } />
															</span>
															<span className="wphb-recipient-name">{ user.name }</span>
														</span>
														<span className="sui-recipient-email">{ user.role ? user.role : user.email }</span>
														<button
															type="button"
															className="sui-button-icon sui-tooltip sui-tooltip-bottom-right"
															data-tooltip={ __( 'Remove recipient', 'wphb' ) }
															onClick={ () => handleRemoveUser( user ) }
														>
															<span className="sui-icon-trash" aria-hidden="true"></span>
														</button>
													</div>
												) ) }
											</div>
										</div>
									) }

									<div className="sui-margin-top">
										<strong> { __( 'Users', 'wphb' ) } </strong>
										<div className="sui-recipients" id="modal-wp-user-list">
											{ loading && <div> { __( 'Loading...', 'wphb' ) } </div> }
											{ availableUsers.length === 0 && ! loading && (
												<div className="sui-notice sui-notice-info sui-margin-top">
													<div className="sui-notice-content">
														<div className="sui-notice-message">
															<span className="sui-notice-icon sui-icon-info" aria-hidden="true"></span>
															<p>{ __( 'No users left to add.', 'wphb' ) }</p>
														</div>
													</div>
												</div>
											) }
											{ availableUsers.map( ( user ) => (
												<div className="sui-recipient" key={ user.email } data-id={ user.id } data-email={ user.email }>
													<span className="recipient-name">
														<span className="subscriber">
															<img src={ user.avatar } alt={ user.email } />
														</span>
														<span className="wphb-recipient-name">{ user.name }</span>
													</span>
													<span className="sui-recipient-email">{ user.role }</span>
													<button type="button" className="sui-button-icon sui-tooltip sui-tooltip-bottom-right" data-tooltip={ __( 'Add recipient', 'wphb' ) } onClick={ () => handleAddUser( user ) }>
														<span className="sui-icon-plus" aria-hidden="true"></span>
													</button>
												</div>
											) ) }
										</div>
									</div>
									{ selected.length === 0 && (
										<div className="sui-notice sui-notice-warning sui-margin-top notifications-recipients-notice">
											<div className="sui-notice-content">
												<div className="sui-notice-message">
													<span className="sui-notice-icon sui-icon-info" aria-hidden="true"></span>
													<p> { __( 'You\'ve not added the users. In order to activate the notification you need to add users first.', 'wphb' ) } </p>
												</div>
											</div>
										</div>
									) }
								</div>
								{ /* Add by Email Tab */ }
								<div role="tabpanel" className={ `sui-tab-content${ activeTab === 'email' ? ' active' : '' }` } hidden={ activeTab !== 'email' }>
									<p className="sui-description">{ __( 'Add credentials below for each recipient.', 'wphb' ) }</p>
									<strong>{ __( 'Invite Recipients', 'wphb' ) }</strong>
									<div className="sui-form-field">
										<label htmlFor="recipient-name" className="sui-label"> { __( 'First name', 'wphb' ) } </label>
										<input
											placeholder="E.g. John"
											id="recipient-name"
											className="sui-form-control"
											value={ name }
											onChange={ ( e ) => setName( e.target.value ) }
										/>
										<span id="error-recipient-name" className="sui-error-message" role="alert"></span>
									</div>
									<div className="sui-form-field">
										<label htmlFor="recipient-email" className="sui-label"> { __( 'Email address', 'wphb' ) } </label>
										<input
											placeholder="E.g John@doe.com"
											id="recipient-email"
											className="sui-form-control"
											value={ email }
											onChange={ ( e ) => setEmail( e.target.value ) }
										/>
										<span id="error-recipient-email" className="sui-error-message" role="alert">{ emailError }</span>
									</div>

									<div className="sui-form-field sui-no-margin-bottom">
										<button type="button" id="add-recipient-button" className="sui-button" aria-live="polite" disabled={ ! name || ! email || loading } onClick={ handleAddEmail }>
											<span className="sui-button-text-default"> { __( 'Add recipient', 'wphb' ) } </span>
											{ loading && (
												<span className="sui-button-text-onload">
													<span className="sui-icon-loader sui-loading" aria-hidden="true"></span>
													{ __( 'Adding recipient', 'wphb' ) }
												</span>
											) }
										</button>
									</div>

									{ selected.length > 0 && (
										<div className="sui-margin-top">
											<strong>{ __( 'Added users', 'wphb' ) }</strong>
											<div className="sui-recipients" id="modal-user-recipients-list">
												{ selected.map( ( user ) => (
													<div className="sui-recipient" key={ user.email } data-id={ user.id } data-email={ user.email }>
														<span className="recipient-name">
															<span className="subscriber">
																<img src={ user.avatar } alt={ user.email } />
															</span>
															<span className="wphb-recipient-name">{ user.name }</span>
														</span>
														<span className="sui-recipient-email">{ user.role ? user.role : user.email }</span>
														<button
															type="button"
															className="sui-button-icon sui-tooltip sui-tooltip-bottom-right"
															data-tooltip={ __( 'Remove recipient', 'wphb' ) }
															onClick={ () => handleRemoveUser( user ) }
														>
															<span className="sui-icon-trash" aria-hidden="true"></span>
														</button>
													</div>
												) ) }
											</div>
										</div>
									) }

									{ selected.length === 0 && (
										<div className="sui-notice sui-notice-warning sui-margin-top notifications-recipients-notice">
											<div className="sui-notice-content">
												<div className="sui-notice-message">
													<span className="sui-notice-icon sui-icon-info" aria-hidden="true"></span>
													<p> { __( 'You\'ve not added the users. In order to activate the notification you need to add users first.', 'wphb' ) } </p>
												</div>
											</div>
										</div>
									) }
								</div>
							</div>
						</div>
					</div>
					<div className="sui-box-footer sui-flatten sui-content-center">
						<button className="sui-button notification-next-buttons" disabled={ selected.length === 0 } onClick={ onClose }>
							{ __( 'Continue', 'wphb' ) }
						</button>
					</div>
				</div>
			</div>
		</div>
	);
}

export default AddRecipientsModal;
