import Scanner from '../utils/scanner';
import Fetcher from '../utils/fetcher';
import { __ } from '@wordpress/i18n';

class ProcessSetupConfiguration extends Scanner {
	constructor( totalSteps, currentStep, props = {} ) {
		super( totalSteps, currentStep );

		const { isMember = false, settings = {}, isNetworkAdmin = false, onProcessSetupFinish } = props;
		this.onProcessSetupFinish = onProcessSetupFinish;

		// Define the base phases
		const basePhases = [];

		// Add Scanning Assets phase only if not network admin
		if ( ! isNetworkAdmin ) {
			basePhases.push( {
				name: __( 'Scanning Assets', 'wphb' ),
				progressStart: 0,
				progressEnd: 20,
				phaseAction: 'wphb_scan_assets',
				cssClass: 'scanning_assets'
			} );
		}

		// Add Critical CSS phase only for member users and if enabled.
		if ( isMember && settings.criticalCSS && ! isNetworkAdmin ) {
			basePhases.push( {
				name: __( 'Generating Critical CSS', 'wphb' ),
				progressStart: 20,
				progressEnd: 60,
				phaseAction: 'wphb_generate_critical_css',
				cssClass: 'generating_critical_css'
			} );
		}

		const remainingPhases = [];

		// Add Page cache phase only if cache is enabled
		if ( settings.cacheEnable ) {
			remainingPhases.push( {
				name: __( 'Generating Page cache for', 'wphb' ),
				progressStart: isMember ? 60 : 20,
				progressEnd: isMember ? 70 : 30,
				phaseAction: 'wphb_generate_page_cache',
				cssClass: 'generating_page_cache'
			} );
		}

		// Add Performance Test phase with adjusted progress ranges
		let performanceTestStart;
		if ( settings.cacheEnable ) {
			performanceTestStart = isMember ? 70 : 30;
		} else {
			performanceTestStart = isMember ? 62 : 22;
		}

		remainingPhases.push( {
			name: __( 'Running Performance Test', 'wphb' ),
			progressStart: performanceTestStart,
			progressEnd: 100,
			phaseAction: 'wphb_performance_test',
			cssClass: 'running_performance_test'
		} );

		this.phases = [ ...basePhases, ...remainingPhases ];

		this.currentPhase = 0;
		this.phaseProgress = 0;
		this.isDestroyed = false;
	}

	/**
	 * Update module status icons in the modal
	 *
	 * @param {number}  currentPhaseIndex - Current phase being processed
	 * @param {boolean} isCompleted       - Whether the current phase is completed
	 */
	updateModuleStatus( currentPhaseIndex, isCompleted = false ) {
		this.phases.forEach( ( phase, index ) => {
			const moduleElement = document.querySelector( `.module-status .${ phase.cssClass } i` );
			const listItem = document.querySelector( `.module-status .${ phase.cssClass }` );

			if ( ! moduleElement || ! listItem ) {
				return;
			}

			// Reset all classes
			moduleElement.className = '';
			listItem.classList.remove( 'processing', 'completed' );

			if ( index < currentPhaseIndex ) {
				// Completed phases
				moduleElement.className = 'sui-icon-check';
				listItem.classList.add( 'completed' );
			} else if ( index === currentPhaseIndex ) {
				// Current phase
				if ( isCompleted ) {
					moduleElement.className = 'sui-icon-check';
					listItem.classList.add( 'completed' );
				} else {
					moduleElement.className = 'sui-icon-loader sui-loading';
					listItem.classList.add( 'processing' );
				}
			} else {
				// Future phases
				moduleElement.className = 'grey-circle';
			}
		} );
	}

	/**
	 * Execute a scan step recursively with four phases.
	 *
	 * @param {number} remainingSteps
	 */
	step( remainingSteps ) {
		// Early exit if scanner was destroyed/cancelled
		if ( this.isDestroyed ) {
			return;
		}

		super.step( remainingSteps );

		if ( this.currentPhase >= this.phases.length ) {
			this.onFinish();
			return;
		}

		// Update progress bar with current phase
		this.updateProgressBar();

		// Update module status to show current phase as processing
		this.updateModuleStatus( this.currentPhase, false );

		// Get the current phase and pass the data using the correct method
		const currentPhase = this.phases[ this.currentPhase ];
		const actionUrl = 'wphb_performance_test' === currentPhase.phaseAction ? 'wphb_performance_run_test' : 'wphb_react_processing_setup_configuration';

		Fetcher.common
			.callWithParams( actionUrl, { phase_action: currentPhase.phaseAction } )
			.then( ( response ) => {
				// Check if cancelled after API response
				if ( this.isDestroyed ) {
					return;
				}

				if ( ! response.finished ) {
					// Increment phase progress but stop at 99% if not finished
					this.phaseProgress = Math.min( this.phaseProgress + 4, 99 );

					// Try again 2 seconds later with cancellation check
					window.setTimeout( () => {
						if ( ! this.isDestroyed ) {
							this.step( remainingSteps );
						}
					}, 2000 );
				} else {
					// Phase completed, mark current phase as completed
					this.updateModuleStatus( this.currentPhase, true );

					// Move to next phase
					this.currentPhase++;
					this.phaseProgress = 0;

					if ( this.currentPhase >= this.phases.length ) {
						this.onFinish();
					} else {
						// Start next phase immediately
						this.step( remainingSteps );
					}
				}
			} )
			.catch( ( error ) => {
				// Silently handle errors if cancelled, otherwise log
				if ( ! this.isDestroyed ) {
					console.error( 'ProcessSetupConfiguration API error:', error );
				}
			} );
	}

	getStatusText( overallProgress ) {
		let statusText = '';

		// Set status text based on progress percentage
		if ( overallProgress >= 90 ) {
			statusText = __( 'Finishing up...', 'wphb' );
		} else if ( overallProgress >= 80 ) {
			statusText = __( 'Almost done...', 'wphb' );
		} else if ( overallProgress >= 60 ) {
			statusText = __( 'Making good progress...', 'wphb' );
		} else if ( overallProgress >= 30 ) {
			statusText = __( 'Moving forward...', 'wphb' );
		} else {
			statusText = __( 'Preparing...', 'wphb' );
		}

		return statusText;
	}

	updateProgressBar() {
		let overallProgress = 0;

		if ( this.currentPhase < this.phases.length ) {
			const phase = this.phases[ this.currentPhase ];

			// Calculate overall progress based on current phase and phase progress
			const phaseContribution = ( phase.progressEnd - phase.progressStart ) * ( this.phaseProgress / 100 );
			overallProgress = phase.progressStart + phaseContribution;

			// For Performance Test phase, ensure it never reaches 100% until actually finished
			if ( 'wphb_performance_test' === phase.phaseAction && this.phaseProgress < 100 ) {
				overallProgress = Math.min( overallProgress, 99 );
			}
		} else {
			overallProgress = 100;
		}

		overallProgress = Math.round( overallProgress );
		// Set status text based on progress percentage
		const statusText = this.getStatusText( overallProgress );

		const progressStatus = document.querySelector(
			'.wphb-performance-scan-modal .sui-progress-state .sui-progress-state-text'
		);

		if ( progressStatus ) {
			progressStatus.innerHTML = statusText;
		}

		const progressTextElement = document.querySelector(
			'.wphb-performance-scan-modal .sui-progress-block .sui-progress-text span'
		);
		if ( progressTextElement ) {
			progressTextElement.innerHTML = overallProgress + '%';
		}

		const progressBarElement = document.querySelector(
			'.wphb-performance-scan-modal .sui-progress-block .sui-progress-bar span'
		);
		if ( progressBarElement ) {
			progressBarElement.style.width = overallProgress + '%';
		}

		if ( 100 === overallProgress ) {
			const loader = document.querySelector(
				'.wphb-performance-scan-modal .sui-progress-block span.sui-icon-loader'
			);
			if ( loader ) {
				loader.classList.remove( 'sui-icon-loader', 'sui-loading' );
				loader.classList.add( 'sui-icon-check' );
			}
		}
	}

	cancel() {
		// Mark as destroyed to prevent any future operations
		this.isDestroyed = true;
		// Call parent cancel method
		super.cancel();
		// Call the callback if provided
		if ( this.onProcessSetupFinish ) {
			this.onProcessSetupFinish();
		}

		window.SUI.closeModal();
	}

	onStart() {
		return Promise.resolve();
	}

	onFinish() {
		// Mark all phases as completed
		this.phases.forEach( ( phase, index ) => {
			this.updateModuleStatus( index, true );
		} );

		// Call the callback if provided
		if ( this.onProcessSetupFinish ) {
			this.onProcessSetupFinish();
		}

		window.SUI.closeModal();
	}
}

export default ProcessSetupConfiguration;
