<?php

namespace Smush\Core\CDN;

use Smush\Core\Controller;
use Smush\Core\Helper;
use Smush\Core\Settings;

class CDN_Settings_Ui_Controller extends Controller {

	/**
	 * @var Settings|null
	 */
	private $settings;

	public function __construct() {
		$this->settings = Settings::get_instance();

		$this->register_filter( 'wp_smush_settings', array( $this, 'register_cdn_settings' ) );
		$this->register_action( 'smush_setting_column_right_inside', array( $this, 'settings_desc' ), 10, 2 );
		$this->register_action( 'smush_setting_column_right_inside', array( $this, 'settings_notice' ), 20 );
	}

	/**
	 * Add settings to settings array.
	 *
	 * @param array $settings Current settings array.
	 *
	 * @return array
	 * @since 3.0
	 */
	public function register_cdn_settings( $settings ) {
		$next_gen_cdn_key = Settings::get_next_gen_cdn_key();

		return array_merge(
			$settings,
			array(
				'background_images' => array(
					'label'       => __( 'Serve background images from the CDN', 'wp-smushit' ),
					'short_label' => __( 'Background Images', 'wp-smushit' ),
					'desc'        => __( 'Where possible we will serve background images declared with CSS directly from the CDN.', 'wp-smushit' ),
				),
				'cdn_dynamic_sizes' => array(
					'label'       => __( 'Add extra image sizes', 'wp-smushit' ),
					'short_label' => __( 'Dynamic Image Sizing', 'wp-smushit' ),
					'desc'        => __( 'Ensure browsers have all the image sizes needed for optimal performance.', 'wp-smushit' ),
				),
				$next_gen_cdn_key   => array(
					'label'       => __( 'Choose Format Settings', 'wp-smushit' ),
					'short_label' => __( 'Next-Gen Conversion', 'wp-smushit' ),
					'desc'        => __( 'Smush can automatically convert and serve your images as WebP or AVIF to compatible browsers.', 'wp-smushit' ),
				),
				'rest_api_support'  => array(
					'label'       => __( 'Enable REST API support', 'wp-smushit' ),
					'short_label' => __( 'REST API', 'wp-smushit' ),
					'desc'        => __( 'Smush can automatically replace image URLs when fetched via REST API endpoints.', 'wp-smushit' ),
				),
			)
		);
	}

	/**
	 * Show additional descriptions for settings.
	 *
	 * @param string $setting_key Setting key.
	 *
	 * @since 3.0
	 */
	public function settings_desc( $setting_key = '' ) {
		if (
			empty( $setting_key )
			|| ! in_array( $setting_key, $this->settings->get_cdn_fields(), true )
			|| Settings::get_next_gen_cdn_key() === $setting_key
		) {
			return;
		}
		?>
		<span class="sui-description sui-toggle-description"
			id="<?php echo esc_attr( 'wp-smush-' . $setting_key . '-desc' ); ?>">
			<?php
			switch ( $setting_key ) {
				case 'cdn_dynamic_sizes':
					esc_html_e( 'Generate extra image sizes on the fly so browsers can display the best one according to device and viewport.', 'wp-smushit' );
					break;
				case 'background_images':
					printf(
						/* translators: %1$s - Open the link <a>, %2$s - Closing link tag */
						esc_html__( 'Note: For this feature to work your theme’s background images must be declared correctly using the default %1$swp_attachment%2$s functions.', 'wp-smushit' ),
						'<a href="https://developer.wordpress.org/reference/functions/wp_get_attachment_image/" target="_blank">',
						'</a>'
					);
					echo '<br>';
					printf(
						/* translators: %1$s - Open the link <a>, %2$s - closing link tag */
						esc_html__( 'For any non-media library uploads, you can still use the %1$sDirectory Smush%2$s feature to compress them, they just won’t be served from the CDN.', 'wp-smushit' ),
						'<a href="' . esc_url( network_admin_url( 'admin.php?page=smush-bulk#directory_smush-settings-row' ) ) . '">',
						'</a>'
					);
					break;
				case 'rest_api_support':
					printf(
						/* translators: %1$s - Open a link <a>, %2$s - closing link tag */
						esc_html__( 'Note: Smush will use the %1$srest_pre_echo_response%2$s hook to filter images in REST API responses.', 'wp-smushit' ),
						'<a href="https://developer.wordpress.org/reference/hooks/rest_pre_echo_response/" target="_blank">',
						'</a>'
					);
					break;
				default:
					break;
			}
			?>
		</span>
		<?php
	}

	/**
	 * Show additional notice under field description for settings.
	 *
	 * @param string $setting_key Setting key.
	 */
	public function settings_notice( $setting_key ) {
		$allowed_setting_keys = array(
			'cdn_dynamic_sizes',
		);
		if (
			empty( $setting_key )
			|| ! in_array( $setting_key, $allowed_setting_keys, true )
		) {
			return;
		}
		?>
		<div style="margin-left:44px"
			id="<?php echo esc_attr( 'wp-smush-' . $setting_key . '-notice' ); ?>">
			<?php
			switch ( $setting_key ) {
				case 'cdn_dynamic_sizes':
					$is_auto_resizing_enabled = $this->settings->is_lazyload_active() &&
												$this->settings->is_auto_resizing_active();
					if ( ! $is_auto_resizing_enabled && $this->settings->has_lazy_preload_page() ) :
					?>
						<div class="sui-upsell-notice" style="margin-top:10px;">
							<div class="sui-notice sui-notice-blue">
								<div class="sui-notice-content">
									<div class="sui-notice-message">
										<span class="sui-notice-icon sui-icon-info sui-md" aria-hidden="true"></span>
										<p>
										<?php
										esc_html_e( 'The automatic resizing feature in the Lazy Load module takes guesswork out of the equation by detecting the container size on page load and making sure the perfect image size is loaded in every situation.', 'wp-smushit' );
										?>
										</p>
										<p><a class="sui-button smush-sui-button-outline smush-sui-button-outline-blue" href="<?php echo esc_url( Helper::get_page_url( 'smush-lazy-preload' ) ); ?>#lazyload-image-resizing-settings-row"><?php esc_html_e( 'Go to Lazy Load', 'wp-smushit' ); ?></a></p>
									</div>
								</div>
							</div>
						</div>
					<?php endif; ?>
					<?php
					break;
				default:
					break;
			}
			?>
		</div>
		<?php
	}
}