<?php

if (!defined('ABSPATH'))	exit;

if (!class_exists('Ph_Ups_Address_Validation_Rest')) {
	class Ph_Ups_Address_Validation_Rest {

		// Class Variables Declaration
		public $residential_check = 0;
		public $destination;
		public $settings;
		public $debug;

		/**
		 * Constructor
		 */
		public function __construct($destination = array(), $settings = array()) {
			$this->destination	= $destination;
			$this->settings		= $settings;
			$this->init();
		}

		/**
		 * Init
		 */
		public function init() {

			$this->debug = $this->settings['debug'];

			$json_request				= $this->get_address_validation_request();
			$json_response				= $this->get_address_validation_response($json_request);
			$matched_addresses			= $this->process_response($json_response);
			$this->residential_check	= $this->process_response_for_residential_commercial($json_response);
		}

		/**
		 * Get Address Validation Request as JSON.
		 * @return string JSON Request.
		 */
		public function get_address_validation_request() {

			$jsonRequest = array();
			$address1 = isset($this->destination['address_1']) ? $this->destination['address_1'] : (isset($this->destination['address']) ? $this->destination['address'] : '');

			if (isset($this->destination['address_2']) && !empty($this->destination['address_2'])) {

				$address1 .= ' ' . $this->destination['address_2'];
			}

			// JSON Request
			$jsonRequest['XAVRequest'] = array(
				'Request'		   => array(
					'TransactionReference' 	=> array(
						'CustomerContext'	=> '** UPS Address Validation **'
					),
					'RequestOption'			=> '3',
				),
				'AddressKeyFormat' => array(
					'AddressLine' 			=> array(
						$address1,
					),
					'PoliticalDivision2'	=> $this->destination['city'],
					'PoliticalDivision1'	=> $this->destination['state'],
					'PostcodePrimaryLow'	=> $this->destination['postcode'],
					'CountryCode'			=> $this->destination['country'],
				),
			);

			return $jsonRequest;
		}

		/**
		 * Get Address Validation Response.
		 * @param string $request JSON request.
		 * @return mixed( bool | string ) Return false on error or JSON Response.
		 */
		public function get_address_validation_response($request) {

			$result = [];
			$send_request = wp_json_encode($request, JSON_UNESCAPED_SLASHES);

			$api_access_details = Ph_Ups_Endpoint_Dispatcher::phGetApiAccessDetails();
			$endpoint = Ph_Ups_Endpoint_Dispatcher::ph_get_enpoint('validated-address');

			$transient_id = 'ph_ups_address_val' . md5(wp_json_encode($request));

			// Fetch cached response.
			$cached_response = get_transient($transient_id);

			// Log the request details.
			Ph_UPS_Woo_Shipping_Common::phAddDebugLog("-------------------- UPS Address Validation Request --------------------", $this->debug);
			Ph_UPS_Woo_Shipping_Common::phAddDebugLog(json_encode($request), $this->debug);

			// Check if cached response exists.
			if ($cached_response) {

				// Log the cached response.
				Ph_UPS_Woo_Shipping_Common::phAddDebugLog("-------------------- UPS Address Validation Cached Response --------------------", $this->debug);
				Ph_UPS_Woo_Shipping_Common::phAddDebugLog($cached_response, $this->debug);

				// Return cached response.
				return $cached_response;
			}

			$result = Ph_Ups_Api_Invoker::phCallApi(
				PH_UPS_Config::PH_UPS_PROXY_API_BASE_URL . $endpoint,
				$api_access_details['token'],
				$send_request
			);

			if ( is_wp_error( $result ) ) {
				
				$error_message = $result->get_error_message();

				// Log the error response.
				Ph_UPS_Woo_Shipping_Common::phAddDebugLog("-------------------- UPS Address Validation WP Error Response --------------------", $this->debug);
				Ph_UPS_Woo_Shipping_Common::phAddDebugLog($error_message, $this->debug);

				return false; // Early return on error.
			}

			$response_body = wp_remote_retrieve_body( $result );

			// Log the response details.
			Ph_UPS_Woo_Shipping_Common::phAddDebugLog("-------------------- UPS Address Validation Response --------------------", $this->debug);
			Ph_UPS_Woo_Shipping_Common::phAddDebugLog($response_body, $this->debug);

			return !empty($error_message) ? false : $response_body;
		}

		/**
		 * Process the JSON response of Address Validation.
		 * @param string $json_response JSON Response.
		 * @return 
		 */
		public function process_response( $json_response ) {

			$json_response = json_decode($json_response, true);
			$response = false;

			if( ! $json_response ) {

				$error_message = "Failed loading JSON : ".print_r( $json_response, true ).PHP_EOL;
				foreach(libxml_get_errors() as $error) {
					$error_message = $error_message . $error->message . PHP_EOL;
				}

				Ph_UPS_Woo_Shipping_Common::debug( "-------------------- UPS Address Validation Response XML Error --------------------". PHP_EOL . $error_message . PHP_EOL , $this->debug, $this->settings['silent_debug'], 'alert' );
			}

			// Match Found
			elseif( isset($json_response['XAVResponse']['ValidAddressIndicator']) ){
				$response = $json_response['XAVResponse']['Candidate'];
				$suggested_address = null;
			} 

			elseif( isset($json_response['XAVResponse']['AmbiguousAddressIndicator']) ){
				$response = $json_response['XAVResponse']['Candidate']; 
			}

			elseif( isset($json_response['XAVResponse']['NoCandidatesIndicator']) ) {
				Ph_UPS_Woo_Shipping_Common::debug( "-------------------- UPS Address Validation Response Message --------------------". PHP_EOL . "No matching Address found." . PHP_EOL , $this->debug, $this->settings['silent_debug'], 'alert' );
			}

			$suggested_option 	= array();
			
			// Show the Suggested address
			if( $response && !is_admin() ) {
				
				$addressLine = isset($this->destination['address_1']) ? $this->destination['address_1'] : ( isset($this->destination['address']) ? $this->destination['address'] : '' );
				
				// With REST API Multiple Addresses are returned hence taking the first one
				$response = isset($response['AddressKeyFormat']) ? $response : ( is_array($response) ? current($response) : $response );

				if( $addressLine != $response['AddressKeyFormat']['AddressLine'] || $this->destination['city'] != $response['AddressKeyFormat']['PoliticalDivision2'] || $this->destination['state'] != $response['AddressKeyFormat']['PoliticalDivision1'] ) {

					if( is_array($response['AddressKeyFormat']['AddressLine']) && isset($response['AddressKeyFormat']['AddressLine'][0]) ) {

						$street_address = $response['AddressKeyFormat']['AddressLine'][0];

						$address_1 		= $street_address;
						$address_2 		= '';

						if( isset($response['AddressKeyFormat']['AddressLine'][1]) ) {

							$street_address .= ', '.$response['AddressKeyFormat']['AddressLine'][1];
							$address_2 		 = $response['AddressKeyFormat']['AddressLine'][1];
						}
					} else {

						$street_address = $response['AddressKeyFormat']['AddressLine'];
						$address_1 		= $street_address;
						$address_2 		= '';
					}
					
					$message = __( 'Suggested Address -', 'ups-woocommerce-shipping' ).'<br/>';
					$message .= __( 'Street Address: ', 'ups-woocommerce-shipping' ).$street_address.'<br/>';
					$message .= __( 'City: ', 'ups-woocommerce-shipping').$response['AddressKeyFormat']['PoliticalDivision2'].'<br/>';
					$message .= __( 'State: ', 'ups-woocommerce-shipping' ). WC()->countries->states['US'][$response['AddressKeyFormat']['PoliticalDivision1']] .'<br/>';
					$message .= __( 'PostCode: ', 'ups-woocommerce-shipping').$response['AddressKeyFormat']['PostcodePrimaryLow'].(isset( $response['AddressKeyFormat']['PostcodeExtendedLow'] ) ? '-' . $response['AddressKeyFormat']['PostcodeExtendedLow'] : '').'<br/>';
					$message .= __( 'Country: ', 'ups-woocommerce-shipping'). WC()->countries->countries[$response['AddressKeyFormat']['CountryCode']];
					
					$message = apply_filters( 'ph_ups_address_validation_message', $message, $response );
					
					$s_address 	= array(
						'country' 		=> $response['AddressKeyFormat']['CountryCode'],
						'state' 		=> $response['AddressKeyFormat']['PoliticalDivision1'],
						'postcode' 		=> $response['AddressKeyFormat']['PostcodePrimaryLow'],
						'city' 			=> $response['AddressKeyFormat']['PoliticalDivision2'],
						'address' 		=> $address_1,
						'address_1' 	=> $address_1,
						'address_2' 	=> $address_2,
					);

					$suggested_option = array(
						'checkout_address' 		=> $this->destination,
						'suggested_address' 	=> $s_address,  
					);

					if( ! empty($message) && $this->settings['suggested_address'] && 'suggested_notice' == $this->settings['suggested_display'] ) {
						
						wc_clear_notices();
						wc_add_notice( $message );
					}
				}
			}

			if( WC() != null && WC()->session != null ){
				
				WC()->session->set('ph_ups_suggested_address_on_checkout', $suggested_option);
			}

			return $response;
		}
		/**
		 * Process the JSON response of Address Validation.
		 * @param string $json_response JSON Response.
		 * @return 
		 */
		public function process_response_for_residential_commercial($json_response) {
			
			$response = json_decode($json_response, true);

			if (!$response) {
				return 0;
			}
			// Match Found
			elseif (isset($response['XAVResponse']['Candidate']) && isset($response['XAVResponse']['Candidate']['AddressClassification'])) {

				$response = $response['XAVResponse']['Candidate']['AddressClassification'];
			} elseif (isset($response['XAVResponse']['AddressClassification'])) {

				$response = $response['XAVResponse']['AddressClassification'];
			}

			// Show the Suggested address
			if ($response  && is_array($response)) {
				$response = isset($response['Code']) ? $response['Code'] : 0;
			}
			return $response;
		}
	}
}
