<?php
/**
 * Handle Woocommerce bundled products on cart and order page while calculate shipping and generate label 
 * Plugin link : https://woocommerce.com/products/product-bundles/
 */

// Skip the bundled products while generating the packages on order page

if( !function_exists('ph_ups_woocommerce_bundle_product_support_on_generate_label') ) {

	function ph_ups_woocommerce_bundle_product_support_on_generate_label( $package, $order) {

		if ( is_a( $order, 'wf_order' ) ) $order = wc_get_order( $order->get_id() );

		if( is_a( $order, 'WC_Order' ) ) {

			$line_items 	= $order->get_items();
			$package 		= array();

			foreach( $line_items as $line_item ) {

				if( is_a($line_item, 'WC_Order_Item_Product') ) {

					$require_shipping 	= $line_item->get_meta('_bundled_item_needs_shipping');

					if( empty($require_shipping) || $require_shipping == 'yes' ) {

						$product 	= $line_item->get_product();

						if( is_a( $product, 'WC_Product') ) {

							if($product->needs_shipping()) {

								// Check whether Price Individually set or not in case of bundle products,
								$priced_individually = $line_item->get_meta('_bundled_item_priced_individually');

								if( $priced_individually == 'yes' ) {

									$product_price = ( (float) $line_item->get_total() ) / $line_item->get_quantity();

									$product->set_price($product_price);
								}
								
								$product_id 	= $product->get_id();

								if( ! isset($package[$product_id])) {

									$package[$product_id] = array(
										'data'		=>	$product,
										'quantity'	=>	$line_item->get_quantity(),
									);

								} else {

									$package[$product_id]['quantity'] += $line_item->get_quantity();
								}
							}

						} else {
							$deleted_products[] = $line_item->get_name();
						}
					}
				}
			}
		}

		if( ! empty($deleted_products) && is_admin() && ! is_ajax() && class_exists('WC_Admin_Meta_Boxes') ) {

			WC_Admin_Meta_Boxes::add_error( __( "UPS Warning! One or more Ordered Products have been deleted from the Order. Please check these Products- ", 'ups-woocommerce-shipping' ).implode( ',', $deleted_products ).'.' );
		}
		
		return $package;
	}
}

add_filter( 'xa_ups_get_customized_package_items_from_order', 'ph_ups_woocommerce_bundle_product_support_on_generate_label', 9, 2 );

// End of skip bundled products and external products while generating the packages

/**
 * Update invoice products to incorporate bundled items for assembled product bundles.
 *
 * This function processes bundled items within an order, ensuring that all child products 
 * are correctly added to the invoice. It enhances the shipment details by including 
 * individual components of bundled products.
 *
 * @param array  $invoice_products  List of existing invoice products.
 * @param object $shipment          Shipment object containing order item data.
 * @param object $order             WooCommerce order object.
 * @param array  $from_address      Origin address for the shipment.
 * @param string $ship_to_country   Destination country code.
 * @param bool   $return_label      Indicates if this is a return shipment.
 * @param float  $total_item_cost   Total cost of all items in the shipment.
 * @param object $rest_admin_obj    REST API admin object with configuration settings.
 * 
 * @return array Updated array of invoice products, including child items from product bundles.
 */
function ph_update_invoice_products_for_product_bundle($invoice_products, $shipment, $order, $from_address, $ship_to_country, $return_label, $total_item_cost, $rest_admin_obj) {

	// Check if the 'include_child_products_for_bundle' setting is not set or is false
    if (!isset($rest_admin_obj->settings['include_child_products_for_bundle']) || !$rest_admin_obj->settings['include_child_products_for_bundle']) {
        
        // If the setting is not enabled, return the original invoice products without modification
        return $invoice_products;
    }

	// Retrieve order object from order ID
	$line_items = $order->get_items();

	$all_product_details = [];

	// Loop through each line item in the order
	foreach ($line_items as $item_id => $line_item) {

		// Ensure the line item is a product
		if (!is_a($line_item, 'WC_Order_Item_Product')) {
			continue;
		}

		$product = $line_item->get_product();

		// Check if the product is a bundle and not virtual
		if (!($product->is_type('bundle') && !$product->is_virtual())) {
			continue;
		}

		// Get bundled items within the product
		$bundled_items = $product->get_bundled_items();

		// Loop through each bundled item
		foreach ($bundled_items as $bundled_item) {

			$bundled_item_data = $bundled_item->item_data;

			// Get invoice details for the bundled item
			$product_details = ph_get_invoice_product_details($line_item, $bundled_item_data, $rest_admin_obj);

			// Check if product details were successfully retrieved and contain an item ID
			if (!(!empty($product_details) && isset($product_details['item_id']))) {
				continue;
			}

			// If the item already exists in the $all_product_details array, update the quantity
			if (isset($all_product_details[$product_details['item_id']])) {
				
				// Increment the unit number by adding the new bundled item's unit count
				$all_product_details[$product_details['item_id']]['Unit']['Number'] += $product_details['product_details']['Unit']['Number'];
			} else {
				// If the item does not exist, add the new product details to the array
				$all_product_details[$product_details['item_id']] = $product_details['product_details'];
			}
		}
	}

	// Merge bundled product details with existing invoice products
	if (!empty($all_product_details)) {
		$invoice_products = array_merge($invoice_products, $all_product_details);
	}

	return $invoice_products;
}

/**
 * Get detailed information for a bundled item to include in the invoice.
 *
 * @param object $line_item WooCommerce order line item.
 * @param array $bundled_item_data Data related to the bundled item.
 * @param object $rest_admin_obj REST API admin object with settings.
 * @return array Product details to be added to the invoice.
 */
function ph_get_invoice_product_details($line_item, $bundled_item_data, $rest_admin_obj) {

	// Determine parent product ID from the bundled item data
	$product_id = $bundled_item_data['product_id'];

	// Initialize variable to store the product variation ID
	$prod_variation_id = '';

	// Retrieve meta data from the line item, specifically looking for '_stamp' which holds product details
	$meta_data = $line_item->get_meta('_stamp', true);

	// Loop through each meta data entry to find a matching product ID
	foreach ($meta_data as $item) {

		// Check if the product ID in the meta data matches the bundled item product ID
		// and verify if a variation ID exists
		if (!empty($item['product_id']) && $item['product_id'] == $product_id && isset($item['variation_id'])) {
			
			// Assign the found variation ID to $prod_variation_id
			$prod_variation_id = $item['variation_id'];
			break;  // Exit the loop once a matching variation is found
		}
	}

	// If a variation ID was found, use it; otherwise, fall back to the parent product ID
	$item_id = $prod_variation_id;

	// Get the product data using the variation ID (if available) or the parent product ID
	$product_data = !empty($item_id) ? wc_get_product($item_id) : wc_get_product($product_id);

	$item_quantity = $line_item->get_quantity();
	$product_weight = $product_data->get_weight();

	// Default weight to 0 if not available
	if (empty($product_weight)) {
		$product_weight = 0;
	}

	// Calculate total quantity based on bundled item quantity
	$item_quantity *= $bundled_item_data['quantity_min'];

	// Calculate product weight in the appropriate unit
	$product_unit_weight = wc_get_weight($product_weight, $rest_admin_obj->settings['weight_unit']);

	// Calculate total weight for the line item
	$product_line_weight = $product_unit_weight * $item_quantity * $bundled_item_data['quantity_min'];
	$product_line_weight = round($product_unit_weight, 1);

	// Get HST code for the product
	$hst = get_post_meta($item_id, '_ph_ups_hst_var', true);

	if (empty($hst)) {
		$hst = get_post_meta($product_id, '_wf_ups_hst', true);
	}

	// Generate product title and handle formatting
	$product_title = ($product_id != $item_id) ? strip_tags($product_data->get_formatted_name()) : $product_data->get_title();

	// Remove special characters if enabled in settings
	if ($rest_admin_obj->settings['remove_special_char_product']) {
		$product_title = preg_replace('/[^A-Za-z0-9-()# ]/', '', $product_title);
		$product_title = htmlspecialchars($product_title);
	}

	// Shorten title if it exceeds 35 characters
	$product_title = (strlen($product_title) >= 35) ? substr($product_title, 0, 32) . '...' : $product_title;

	// Determine product price based on admin settings
	if ($rest_admin_obj->settings['invoice_commodity_value'] == 'discount_price') {
		$product_price = $line_item->get_total() / $line_item->get_quantity();
	} else if ($rest_admin_obj->settings['invoice_commodity_value'] == 'declared_price') {
		$custom_declared_value = get_post_meta($item_id, '_ph_ups_custom_declared_value_var', true);

		if (empty($custom_declared_value)) {
			$custom_declared_value = get_post_meta($product_id, '_wf_ups_custom_declared_value', true);
		}
		$product_price = !empty($custom_declared_value) ? $custom_declared_value : $product_data->get_price();
	} else {
		$product_price = $product_data->get_price();
	}

	// Use fixed product price if the calculated price is zero
	$product_price = ($product_price == 0) ? $rest_admin_obj->settings['fixedProductPrice'] : $product_price;

	// Determine the country of manufacture
	$country_of_manufacture = get_post_meta($product_id, '_ph_ups_manufacture_country', true);

	if (empty($country_of_manufacture)) {
		$country_of_manufacture = $rest_admin_obj->settings['origin_country'];
	}

	// Generate invoice description
	$invoice_description = $rest_admin_obj->ph_get_commercial_invoice_description($product_id, $product_title, $item_id);

	$item_id = !empty($item_id) ? $item_id : $product_id;

	// Prepare product details for invoice
	$product_details = array(
		'item_id'			=> $item_id,
		'product_details' 	=> array(
			'Unit' => array(
				'Number' => $item_quantity,
				'UnitOfMeasurement' => array('Code' => $rest_admin_obj->settings['invoice_unit_of_measure']),
				'Value' => round($product_price, 2),
			),
			'OriginCountryCode' => $country_of_manufacture,
			'NumberOfPackagesPerCommodity' => '1',
			'ProductWeight' => array(
				'UnitOfMeasurement' => array(
					'Code' => $rest_admin_obj->settings['weight_unit'],
				),
				'Weight' => $product_line_weight,
			),
			'CommodityCode' => $hst,
			'Description' => $invoice_description,
		)
	);

	return $product_details;
}

// Filter to update invoice items for commercial invoices
add_filter('ph_ups_shipment_invoice_product_details', 'ph_update_invoice_products_for_product_bundle', 9, 8);
