<?php

if ( ! class_exists( 'PH_UPS_Weight_Packing_Util' ) ) {

	class PH_UPS_Weight_Packing_Util {

		/**
		 * Packs items into boxes based on maximum allowed weight and box weight.
		 *
		 * @param array $items         Array of items to pack.
		 * @param float $max_weight    Maximum allowable weight for a single box (excluding box weight).
		 * @param float $box_weight    Weight of the empty box.
		 * @param float $max_quantity  Maximum allowable quantity for a single box.

		 * @return PH_UPS_Weight_Packing_Result  Object containing packed boxes and unpacked items.
		 */
		public function pack_items_into_weight_box( $items, $max_weight, $box_weight, $max_quantity ) {

			$boxes    = array();
			$unpacked = array();

			foreach ( $items as $item ) {

				$fitted      = false;
				$item_weight = $item['weight'];

				foreach ( $boxes as $box_key  => $box ) {
					if ( ( ( $max_weight - $box['weight'] ) >= $item_weight ) && ( empty( $max_quantity ) || ( $max_quantity > $box['quantity'] ) ) ) {
						$boxes[ $box_key ]['weight']   = $boxes[ $box_key ]['weight'] + $item_weight;
						$boxes[ $box_key ]['quantity'] = ! empty( $boxes[ $box_key ]['quantity'] ) ? ++$boxes[ $box_key ]['quantity'] : 1;
						$boxes[ $box_key ]['items'][]  = $item['data'];
						$fitted      				   = true;
						break;
					}
				}

				if ( ! $fitted ) {
					
					if ( $item_weight + $box_weight <= $max_weight ) {
						
						$boxes[] = array(
							'weight'   => $item_weight + $box_weight,
							'quantity' => 1,
							'items'    => array( $item['data'] ),
						);
					} else {

						$unpacked[] = array(
							'weight' => $item_weight,
							'items'  => array( $item['data'] ),
						);
					}
				}
			}
			
			$result = new PH_UPS_Weight_Packing_Result();
			
			$result->set_packed_boxes( $boxes );
			$result->set_unpacked_items( $unpacked );

			return $result;
		}

		/**
		 * Packs all items into one or multiple boxes based on max quantity per box.
		 *
		 * If $max_quantity is defined, a new box is started once the limit is reached.
		 * Each box includes its total weight (including box weight) and items packed inside.
		 *
		 * @param array $items        Array of items, where each item contains 'weight' and 'data'.
		 * @param float $box_weight   The weight of an empty box.
		 * @param int   $max_quantity The maximum number of items allowed per box. If empty, all items go into one box.
		 *
		 * @return PH_UPS_Weight_Packing_Result The result object containing packed boxes.
		 */
		public function pack_all_items_into_one_box( $items, $box_weight, $max_quantity ) {

			$boxes 	      = array();
			$box_items    = array();
			$total_weight = 0;
			$count        = 0; // Start from 0 and increase.
		
			foreach ( $items as $item ) {

				$total_weight += $item['weight'];
				$box_items[]   = $item['data'];
				$count++;
		
				// If max_quantity is set and reached, create a new box.
				if ( ! empty( $max_quantity ) && $count === (int)$max_quantity ) {
					$boxes[] = array(
						'weight' => $total_weight + $box_weight,
						'items'  => $box_items,
					);
		
					// Reset for the next box.
					$count 		  = 0;
					$total_weight = 0;
					$box_items 	  = array();
				}
			}
		
			// Add the last box if it contains any items / add all the items to one box when max quantity is empty.
			if ( ! empty( $box_items ) ) {
				$boxes[] = array(
					'weight' => $total_weight + $box_weight,
					'items'  => $box_items,
				);
			}
		
			$result = new PH_UPS_Weight_Packing_Result();
			$result->set_packed_boxes( $boxes );
		
			return $result;
		}		
	}
}