<?php

require 'class-ph-ups-help-and-support.php';

/**
 * WF_Shipping_UPS class.
 *
 * @extends WC_Shipping_Method
 */
class WF_Shipping_UPS extends WC_Shipping_Method {

	public $mode     = 'volume_based';
	public $vendorId = null;
	public $services;

	// Packaging not offered at this time: 00 = UNKNOWN, 30 = Pallet, 04 = Pak
	// Code 21 = Express box is valid code, but doesn't have dimensions
	// References:
	// http://www.ups.com/content/us/en/resources/ship/packaging/supplies/envelopes.html
	// http://www.ups.com/content/us/en/resources/ship/packaging/supplies/paks.html
	// http://www.ups.com/content/us/en/resources/ship/packaging/supplies/boxes.html
	private $packaging = array(
		'A_UPS_LETTER'         => array(
			'name'        => 'UPS Letter',
			'code'        => '01',
			'length'      => '12.5',
			'width'       => '9.5',
			'height'      => '0.25',
			'weight'      => '0.5',
			'box_enabled' => true,
		),
		'B_TUBE'               => array(
			'name'        => 'Tube',
			'code'        => '03',
			'length'      => '38',
			'width'       => '6',
			'height'      => '6',
			'weight'      => '100',
			'box_enabled' => true,
		),
		'C_PAK'                => array(
			'name'        => 'PAK',
			'code'        => '04',
			'length'      => '17',
			'width'       => '13',
			'height'      => '1',
			'weight'      => '100',
			'box_enabled' => true,
		),
		'D_25KG_BOX'           => array(
			'name'        => '25KG Box',
			'code'        => '24',
			'length'      => '19.375',
			'width'       => '17.375',
			'height'      => '14',
			'weight'      => '25',
			'box_enabled' => true,
		),
		'E_10KG_BOX'           => array(
			'name'        => '10KG Box',
			'code'        => '25',
			'length'      => '16.5',
			'width'       => '13.25',
			'height'      => '10.75',
			'weight'      => '10',
			'box_enabled' => true,
		),
		'F_SMALL_EXPRESS_BOX'  => array(
			'name'        => 'Small Express Box',
			'code'        => '2a',
			'length'      => '13',
			'width'       => '11',
			'height'      => '2',
			'weight'      => '100',
			'box_enabled' => true,
		),
		'G_MEDIUM_EXPRESS_BOX' => array(
			'name'        => 'Medium Express Box',
			'code'        => '2b',
			'length'      => '15',
			'width'       => '11',
			'height'      => '3',
			'weight'      => '100',
			'box_enabled' => true,
		),
		'H_LARGE_EXPRESS_BOX'  => array(
			'name'        => 'Large Express Box',
			'code'        => '2c',
			'length'      => '18',
			'width'       => '13',
			'height'      => '3',
			'weight'      => '30',
			'box_enabled' => true,
		),
	);

	/**
	 * UPS Id
	 */
	public $id;
	/**
	 * General Settings
	 */
	public $settings, $import_control_settings;
	public $ups_tradability, $skip_products, $tax_indicator, $negotiated, $ups_user_name, $ups_display_name, $phone_number, $residential, $isc, $soap_available, $accesspoint_locator, $address_validation, $xa_show_all, $min_order_amount_for_insurance, $enable_freight, $upsSimpleRate, $enable_density_based_rating;
	public $countries, $enabled, $title, $cheapest_rate_title, $availability, $method_title, $method_description, $rate_caching, $customer_classification, $offer_rates, $fallback, $currency_type, $conversion_rate, $min_amount, $fixedProductPrice, $insuredvalue, $hst_lc;
	/**
	 * Packing
	 */
	public $packing_method, $ups_packaging, $weight_packing_process;
	/**
	 * Order
	 */
	public $order;
	/**
	 * Est Delivery
	 */
	public $show_est_delivery, $wp_date_time_format, $estimated_delivery_text, $shipTimeAdjustment, $current_wp_time, $current_wp_time_hour_minute;
	/**
	 * Debug
	 */
	public $debug, $silent_debug;
	/**
	 * UPS Account Details Variables
	 */
	public $user_id, $access_key, $password, $shipper_number;
	/**
	 * Label Settings
	 */
	public $disble_ups_print_label, $print_label_type, $show_label_in_browser, $ship_from_address, $disble_shipment_tracking;
	/**
	 * Pickup
	 */
	public $pickup;
	/**
	 * Boxes
	 */
	public $boxes, $simpleRateBoxes;
	/**
	 * Weight and Dimensions Variables
	 */
	public $exclude_box_weight, $stack_to_volume, $density_length, $density_width, $density_height, $box_max_weight, $box_weight, $min_weight_limit, $max_weight_limit, $min_shipping_cost, $max_shipping_cost;

	public $eu_country_cod_type;

	/**
	 * Package
	 */
	public $current_package_items_and_quantity;
	/**
	 * Unit Variables
	 */
	public $savedMetrics, $wc_weight_unit, $units, $weight_unit, $dim_unit, $density_unit, $uom;
	/**
	 * COD Variables
	 */
	public $cod, $cod_total, $cod_enable;
	/**
	 * Address Variables
	 */
	public $freight_class, $destination, $origin_addressline, $origin_addressline_2, $origin_city, $origin_postcode, $origin_country_state, $origin_country, $ship_from_address_different_from_shipper, $ship_from_addressline, $ship_from_addressline_2, $ship_from_city, $ship_from_postcode, $ship_from_country_state, $ship_from_country, $ship_from_state, $origin_state, $ship_from_custom_state, $origin_custom_state;
	/**
	 * Custom Fields Variables
	 */
	public $ph_delivery_confirmation, $saturday_delivery, $rate_conversion, $ph_restricted_article, $ph_diog, $ph_perishable, $ph_alcoholic, $ph_plantsindicator, $ph_seedsindicator, $ph_specialindicator, $ph_tobaccoindicator, $is_hazmat_product;
	/**
	 * Services
	 */
	public $custom_services, $service_code, $ordered_services;
	/**
	 * Access Point
	 */
	public $ph_ups_selected_access_point_details;
	/**
	 * Delivery Confirmation
	 */
	public $international_delivery_confirmation_applicable;
	/**
	 * Pickup Variables
	 */
	public $pickup_date, $pickup_time;

	/**
	 * __construct function.
	 *
	 * @access public
	 * @return void
	 */
	public function __construct( $order = null ) {
		if ( $order ) {
			$this->order = $order;
		}

		$plugin_config = ph_wc_ups_plugin_configuration();

		$this->id                 = $plugin_config['id'];
		$this->method_title       = $plugin_config['method_title'];
		$this->method_description = $plugin_config['method_description'];

		// WF: Load UPS Settings.
		$ups_settings = get_option( 'woocommerce_' . WF_UPS_ID . '_settings', null );

		$this->wc_weight_unit = get_option( 'woocommerce_weight_unit' );
		$ups_settings         = apply_filters( 'ph_ups_plugin_settings', $ups_settings, $order );

		$this->init();

		// Add Estimated delivery to cart rates
		if ( $this->show_est_delivery ) {
			add_filter( 'woocommerce_cart_shipping_method_full_label', array( $this, 'ph_ups_add_delivery_time' ), 10, 2 );
		}

		// Filter to update the chosen shipping method based on session data
		add_filter( 'woocommerce_shipping_chosen_method', array( $this, 'ph_ups_update_shipping_chosen_method' ), 20, 2 );
	}

	/**
	 * Update the chosen shipping method in WooCommerce using session data.
	 *
	 * @param string $chosen_method The default chosen shipping method.
	 * @param array  $available_methods The available shipping methods for the package.
	 *
	 * @return string Updated chosen shipping method if a valid session method exists, otherwise the original.
	 */
	public function ph_ups_update_shipping_chosen_method( $chosen_method, $available_methods ) {

		// Check if a previously chosen shipping method exists in session and is available
		$session_methods = WC()->session->get( 'chosen_shipping_methods' );

		$session_method = ! empty( $session_methods ) && is_array( $session_methods ) ? reset( $session_methods ) : $session_methods;

		if ( ! empty( $session_method ) && isset( $available_methods[ $session_method ] ) ) {
			return $session_method;
		}

		return $chosen_method;
	}

	/**
	 * Append the estimated delivery time to the shipping method label.
	 *
	 * @param string $label The current shipping method label.
	 * @param object $method The shipping method object.
	 * @return string The modified shipping method label with the estimated delivery time appended.
	 */
	public function ph_ups_add_delivery_time( $label, $method ) {
		return PH_WC_UPS_Common_Utils::ph_update_delivery_time( $label, $method, $this->wp_date_time_format, $this->estimated_delivery_text );
	}

	/**
	 * init function.
	 *
	 * @access public
	 * @return void
	 */
	private function init() {

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		$this->settings = apply_filters( 'ph_ups_plugin_settings', $this->settings, '' );

		$this->isc = isset( $this->settings['international_special_commodities'] ) && ! empty( $this->settings['international_special_commodities'] ) && $this->settings['international_special_commodities'] == 'yes' ? true : false;

		$this->soap_available = Ph_UPS_Woo_Shipping_Common::is_soap_available() ? true : false;

		// Define user set variables.
		$this->mode                = isset( $this->settings['packing_algorithm'] ) ? $this->settings['packing_algorithm'] : 'volume_based';
		$this->exclude_box_weight  = ( isset( $this->settings['exclude_box_weight'] ) && $this->settings['exclude_box_weight'] == 'yes' ) ? true : false;
		$this->stack_to_volume     = ( isset( $this->settings['stack_to_volume'] ) && $this->settings['stack_to_volume'] == 'yes' ) ? true : false;
		$this->enabled             = isset( $this->settings['enabled'] ) ? $this->settings['enabled'] : $this->enabled;
		$this->title               = isset( $this->settings['title'] ) ? $this->settings['title'] : $this->method_title;
		$this->cheapest_rate_title = isset( $this->settings['title'] ) ? $this->settings['title'] : null;
		$this->availability        = isset( $this->settings['availability'] ) ? $this->settings['availability'] : 'all';
		$this->countries           = isset( $this->settings['countries'] ) ? $this->settings['countries'] : array();

		// API Settings
		$this->user_id = isset( $this->settings['user_id'] ) ? $this->settings['user_id'] : '';

		// WF: Print Label - Start
		$this->disble_ups_print_label = isset( $this->settings['disble_ups_print_label'] ) ? $this->settings['disble_ups_print_label'] : '';
		$this->print_label_type       = isset( $this->settings['print_label_type'] ) ? $this->settings['print_label_type'] : 'gif';
		$this->show_label_in_browser  = isset( $this->settings['show_label_in_browser'] ) ? $this->settings['show_label_in_browser'] : 'no';
		$this->ship_from_address      = isset( $this->settings['ship_from_address'] ) ? $this->settings['ship_from_address'] : 'origin_address';

		$this->ups_user_name    = isset( $this->settings['ups_user_name'] ) ? $this->settings['ups_user_name'] : '';
		$this->ups_display_name = isset( $this->settings['ups_display_name'] ) ? $this->settings['ups_display_name'] : '';
		$this->phone_number     = isset( $this->settings['phone_number'] ) ? $this->settings['phone_number'] : '';
		// WF: Print Label - End

		$this->user_id        = isset( $this->settings['user_id'] ) ? $this->settings['user_id'] : '';
		$this->password       = isset( $this->settings['password'] ) ? $this->settings['password'] : '';
		$this->access_key     = isset( $this->settings['access_key'] ) ? $this->settings['access_key'] : '';
		$this->shipper_number = isset( $this->settings['shipper_number'] ) ? $this->settings['shipper_number'] : '';

		$this->negotiated           = isset( $this->settings['negotiated'] ) && $this->settings['negotiated'] == 'yes' ? true : false;
		$this->tax_indicator        = isset( $this->settings['tax_indicator'] ) && $this->settings['tax_indicator'] == 'yes' ? true : false;
		$this->origin_addressline   = isset( $this->settings['origin_addressline'] ) ? $this->settings['origin_addressline'] : '';
		$this->origin_addressline_2 = isset( $this->settings['origin_addressline_2'] ) ? $this->settings['origin_addressline_2'] : '';
		$this->origin_city          = isset( $this->settings['origin_city'] ) ? $this->settings['origin_city'] : '';
		$this->origin_postcode      = isset( $this->settings['origin_postcode'] ) ? $this->settings['origin_postcode'] : '';
		$this->origin_country_state = isset( $this->settings['origin_country_state'] ) ? $this->settings['origin_country_state'] : '';
		$this->debug                = isset( $this->settings['debug'] ) && $this->settings['debug'] == 'yes' ? true : false;
		$this->silent_debug         = isset( $this->settings['silent_debug'] ) && $this->settings['silent_debug'] == 'yes' ? true : false;

		// Estimated delivery : Start
		$this->show_est_delivery       = ( isset( $this->settings['enable_estimated_delivery'] ) && $this->settings['enable_estimated_delivery'] == 'yes' ) ? true : false;
		$this->estimated_delivery_text = ! empty( $this->settings['estimated_delivery_text'] ) ? $this->settings['estimated_delivery_text'] : null;
		$this->shipTimeAdjustment      = ( isset( $this->settings['ship_time_adjustment'] ) && ! empty( $this->settings['ship_time_adjustment'] ) ) ? $this->settings['ship_time_adjustment'] : '';

		$this->fixedProductPrice = ( isset( $this->settings['fixed_product_price'] ) && ! empty( $this->settings['fixed_product_price'] ) ) ? $this->settings['fixed_product_price'] : 1;

		if ( $this->show_est_delivery ) {
			if ( empty( $this->current_wp_time ) ) {
				$current_time          = current_time( 'Y-m-d H:i:s' );
				$this->current_wp_time = date_create( $current_time );
			}
			if ( empty( $this->wp_date_time_format ) ) {
				$this->wp_date_time_format = Ph_UPS_Woo_Shipping_Common::get_wordpress_date_format() . ' ' . Ph_UPS_Woo_Shipping_Common::get_wordpress_time_format();
			}
		}
		// Estimated delivery : End

		$this->rate_caching = ( isset( $this->settings['ups_rate_caching'] ) && ! empty( $this->settings['ups_rate_caching'] ) ) ? $this->settings['ups_rate_caching'] : '24';

		// Pickup and Destination
		$this->pickup                  = isset( $this->settings['pickup'] ) ? $this->settings['pickup'] : '01';
		$this->customer_classification = isset( $this->settings['customer_classification'] ) ? $this->settings['customer_classification'] : '99';

		// Services and Packaging
		$this->offer_rates                    = isset( $this->settings['offer_rates'] ) ? $this->settings['offer_rates'] : 'all';
		$this->fallback                       = ! empty( $this->settings['fallback'] ) ? $this->settings['fallback'] : '';
		$this->currency_type                  = ! empty( $this->settings['currency_type'] ) ? $this->settings['currency_type'] : get_woocommerce_currency();
		$this->conversion_rate                = ! empty( $this->settings['conversion_rate'] ) ? $this->settings['conversion_rate'] : 1;
		$this->packing_method                 = isset( $this->settings['packing_method'] ) ? $this->settings['packing_method'] : 'per_item';
		$this->ups_packaging                  = isset( $this->settings['ups_packaging'] ) ? $this->settings['ups_packaging'] : array();
		$this->custom_services                = isset( $this->settings['services'] ) ? $this->settings['services'] : array();
		$this->boxes                          = isset( $this->settings['boxes'] ) ? $this->settings['boxes'] : array();
		$this->insuredvalue                   = isset( $this->settings['insuredvalue'] ) && $this->settings['insuredvalue'] == 'yes' ? true : false;
		$this->min_order_amount_for_insurance = ! empty( $this->settings['min_order_amount_for_insurance'] ) ? $this->settings['min_order_amount_for_insurance'] : 0;

		$this->upsSimpleRate = isset( $this->settings['ups_simple_rate'] ) && $this->settings['ups_simple_rate'] == 'yes' ? true : false;

		$this->enable_density_based_rating = ( isset( $this->settings['enable_density_based_rating'] ) && $this->settings['enable_density_based_rating'] == 'yes' ) ? true : false;
		$this->density_length              = ( isset( $this->settings['density_length'] ) && ! empty( $this->settings['density_length'] ) ) ? $this->settings['density_length'] : 0;
		$this->density_width               = ( isset( $this->settings['density_width'] ) && ! empty( $this->settings['density_width'] ) ) ? $this->settings['density_width'] : 0;
		$this->density_height              = ( isset( $this->settings['density_height'] ) && ! empty( $this->settings['density_height'] ) ) ? $this->settings['density_height'] : 0;

		$this->freight_class = ! empty( $this->settings['freight_class'] ) ? $this->settings['freight_class'] : 50;

		$this->box_weight = ! empty( $this->settings['box_weight'] ) ? $this->settings['box_weight'] : 0;

		$this->service_code = '';
		$this->min_amount   = isset( $this->settings['min_amount'] ) ? $this->settings['min_amount'] : 0;
		// $this->ground_freight    = isset( $this->settings['ground_freight'] ) && $this->settings['ground_freight'] == 'yes' ? true : false;

		// Units
		$this->units = isset( $this->settings['units'] ) ? $this->settings['units'] : 'imperial';

		$this->savedMetrics = $this->get_option( 'units' );

		if ( $this->units == 'metric' ) {
			$this->weight_unit     = 'KGS';
			$this->dim_unit        = 'CM';
			$this->simpleRateBoxes = PH_WC_UPS_Constants::UPS_SIMPLE_RATE_BOXES_IN_CMS;
			$this->packaging       = PH_WC_UPS_Constants::UPS_DEFAULT_BOXES_IN_CMS;
		} else {
			$this->weight_unit     = 'LBS';
			$this->dim_unit        = 'IN';
			$this->simpleRateBoxes = PH_WC_UPS_Constants::UPS_SIMPLE_RATE_BOXES_IN_INCHES;
			$this->packaging       = PH_WC_UPS_Constants::UPS_DEFAULT_BOXES_IN_INCHES;
		}

		$this->uom = ( $this->units == 'imperial' ) ? 'LB' : 'KG';

		// Advanced Settings
		$this->accesspoint_locator = ( isset( $this->settings['accesspoint_locator'] ) && $this->settings['accesspoint_locator'] == 'yes' ) ? true : false;

		$this->xa_show_all = isset( $this->settings['xa_show_all'] ) && $this->settings['xa_show_all'] == 'yes' ? true : false;

		$this->origin_country = PH_WC_UPS_Common_Utils::ph_get_origin_country_and_state( $this->settings, 'country' );
		$this->origin_state   = PH_WC_UPS_Common_Utils::ph_get_origin_country_and_state( $this->settings, 'state' );

		// COD selected
		$this->cod                 = false;
		$this->cod_total           = 0;
		$this->cod_enable          = ( isset( $this->settings['cod_enable'] ) && ! empty( $this->settings['cod_enable'] ) && $this->settings['cod_enable'] == 'yes' ) ? true : false;
		$this->eu_country_cod_type = isset( $this->settings['eu_country_cod_type'] ) && ! empty( $this->settings['eu_country_cod_type'] ) ? $this->settings['eu_country_cod_type'] : 9;

		$this->services = PH_WC_UPS_Common_Utils::get_services_based_on_origin( $this->origin_country );

		// Different Ship From Address
		$this->ship_from_country_state = ! empty( $this->settings['ship_from_country_state'] ) ? $this->settings['ship_from_country_state'] : null;

		if ( empty( $this->ship_from_country_state ) ) {
			$this->ship_from_country = $this->origin_country_state;     // By Default Origin Country
			$this->ship_from_state   = $this->origin_state;             // By Default Origin State
		} elseif ( strstr( $this->ship_from_country_state, ':' ) ) {
			list($this->ship_from_country, $this->ship_from_state) = explode( ':', $this->ship_from_country_state );
		} else {
			$this->ship_from_country = $this->ship_from_country_state;
			$this->ship_from_state   = '';
		}

		$this->ship_from_custom_state = ! empty( $this->settings['ship_from_custom_state'] ) ? $this->settings['ship_from_custom_state'] : $this->ship_from_state;

		$this->skip_products            = ! empty( $this->settings['skip_products'] ) ? $this->settings['skip_products'] : array();
		$this->min_weight_limit         = ! empty( $this->settings['min_weight_limit'] ) ? (float) $this->settings['min_weight_limit'] : null;
		$this->max_weight_limit         = ! empty( $this->settings['max_weight_limit'] ) ? (float) $this->settings['max_weight_limit'] : null;
		$this->min_shipping_cost        = ! empty( $this->settings['min_shipping_cost'] ) ? (float) $this->settings['min_shipping_cost'] : null;
		$this->max_shipping_cost        = ! empty( $this->settings['max_shipping_cost'] ) ? (float) $this->settings['max_shipping_cost'] : null;
		$this->ph_delivery_confirmation = isset( $this->settings['ph_delivery_confirmation'] ) && ! empty( $this->settings['ph_delivery_confirmation'] ) ? $this->settings['ph_delivery_confirmation'] : 0;
		$this->saturday_delivery        = isset( $this->settings['saturday_delivery'] ) && ! empty( $this->settings['saturday_delivery'] ) ? $this->settings['saturday_delivery'] : '';
		$this->import_control_settings  = ( isset( $this->settings['import_control_settings'] ) && ! empty( $this->settings['import_control_settings'] ) ) ? $this->settings['import_control_settings'] : '';

		if ( ! empty( $this->conversion_rate ) ) {
			$this->rate_conversion = $this->conversion_rate; // For Returned Rate Conversion to Default Currency
			$this->conversion_rate = apply_filters( 'ph_ups_currency_conversion_rate', $this->conversion_rate, $this->currency_type );   // Multicurrency
		}

		add_action( 'woocommerce_update_options_shipping_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_update_options_shipping_' . $this->id, array( $this, 'clear_transients' ) );

		if ( isset( $_GET['ph_ups_reset_boxes'] ) ) {
			add_action( 'admin_init', array( $this, 'ph_ups_reset_boxes' ) );
		}
	}

	/**
	 * Clear Transients
	 *
	 * @access public
	 * @return void
	 */
	public function clear_transients() {
		global $wpdb;

		$wpdb->query( "DELETE FROM `$wpdb->options` WHERE `option_name` LIKE ('_transient_ups_quote_%') OR `option_name` LIKE ('_transient_timeout_ups_quote_%')" );
	}

	/**
	 * Reset boxes
	 */
	public function ph_ups_reset_boxes() {

		$this->update_option( 'boxes', array() );
		$this->update_option( 'ups_packaging', array() );

		$boxesToSave = array();

		foreach ( $this->packaging as $key => $box ) {

			$boxesToSave[ $key ] = array(

				'boxes_name'   => $box['name'],
				'outer_length' => $box['length'],
				'outer_width'  => $box['width'],
				'outer_height' => $box['height'],
				'inner_length' => $box['length'],
				'inner_width'  => $box['width'],
				'inner_height' => $box['height'],
				'box_weight'   => 0,
				'max_weight'   => $box['weight'],
				'max_quantity' => 0,
				'box_enabled'  => $box['box_enabled'],
			);
		}

		foreach ( $this->simpleRateBoxes as $key => $box ) {

			$boxesToSave[ $key ] = array(

				'boxes_name'   => $box['name'],
				'outer_length' => $box['length'],
				'outer_width'  => $box['width'],
				'outer_height' => $box['height'],
				'inner_length' => $box['length'],
				'inner_width'  => $box['width'],
				'inner_height' => $box['height'],
				'box_weight'   => $box['box_weight'],
				'max_weight'   => $box['max_weight'],
				'max_quantity' => 0,
				'box_enabled'  => $box['box_enabled'],
			);
		}

		$this->update_option( 'boxes', $boxesToSave );

		wp_redirect( admin_url( 'admin.php?page=wc-settings&tab=shipping&section=wf_shipping_ups&subtab=packaging' ) );
	}

	/**
	 * Environment Check
	 *
	 * @access public
	 * @return void
	 */
	private function environment_check() {

		$error_message = '';

		// WF: Print Label - Start
		// Check for UPS User Name
		if ( ! $this->ups_user_name && $this->enabled == 'yes' ) {
			$error_message .= '<p>' . __( 'UPS is enabled, but Your Name has not been set.', 'ups-woocommerce-shipping' ) . '</p>';
		}
		// WF: Print Label - End

		// Check for Origin Postcode
		if ( ! $this->origin_postcode && $this->enabled == 'yes' ) {
			$error_message .= '<p>' . __( 'UPS is enabled, but the origin postcode has not been set.', 'ups-woocommerce-shipping' ) . '</p>';
		}

		// Check for Origin country
		if ( ! $this->origin_country_state && $this->enabled == 'yes' ) {
			$error_message .= '<p>' . __( 'UPS is enabled, but the origin country/state has not been set.', 'ups-woocommerce-shipping' ) . '</p>';
		}

		// If user has selected to pack into boxes,
		// Check if at least one UPS packaging is chosen, or a custom box is defined
		if ( ( $this->packing_method == 'box_packing' ) && ( $this->enabled == 'yes' ) ) {

			if ( empty( $this->ups_packaging ) && empty( $this->boxes ) ) {
				$error_message .= '<p>' . __( 'UPS is enabled, and Parcel Packing Method is set to \'Pack into boxes\', but no UPS Packaging is selected and there are no custom boxes defined. Items will be packed individually.', 'ups-woocommerce-shipping' ) . '</p>';
			}
		}

		// Check for at least one service enabled
		$ctr = 0;

		if ( isset( $this->custom_services ) && is_array( $this->custom_services ) ) {

			foreach ( $this->custom_services as $key => $values ) {

				if ( $values['enabled'] == 1 ) {

					++$ctr;
				}
			}
		}

		if ( ( $ctr == 0 ) && $this->enabled == 'yes' ) {

			$error_message .= '<p>' . __( 'UPS is enabled, but there are no services enabled.', 'ups-woocommerce-shipping' ) . '</p>';
		}

		if ( ! $error_message == '' ) {

			echo '<div class="error">';
			echo $error_message;
			echo '</div>';
		}
	}

	/**
	 * Admin Options.
	 *
	 * @access public
	 * @return void
	 */
	public function admin_options() {

		echo '<h3>' . ( ! empty( $this->method_title ) ? $this->method_title : __( 'Settings', 'ups-woocommerce-shipping' ) ) . '</h3>';
		echo ( ! empty( $this->method_description ) ) ? wpautop( $this->method_description ) : '';

		// Check users environment supports this method
		$this->environment_check();

		// Show settings
		?>
		<table class="form-table">
			<?php $this->generate_settings_html(); ?>
		</table>
		<?php
	}

	/**
	 * Generate Registration Banner
	 *
	 * @access public
	 * @return void
	 */
	public function generate_ups_registration_banner_html() {

		ob_start();

		if ( ! Ph_UPS_Woo_Shipping_Common::phIsNewRegistration() && empty( $this->access_key ) ) {
			?>

			<div class="ph_registration_banner">

				<div id='ph_ups_close_banner'>[X]</div>

				<div class="ph_ups_box_1">

					<img src="<?php echo plugins_url( 'ups-woocommerce-shipping' ) . '/resources/images/PH_Logo.png'; ?>">

					<h4><?php _e( 'WooCommerce UPS Shipping Plugin with Print Label', 'ups-woocommerce-shipping' ); ?></h4>

				</div>

				<div class="ph_ups_box_2">

					<ul>
						<li style='color:red;'><strong><?php _e( "Don't have UPS Access Keys? <a href='" . admin_url( '/admin.php?page=ph_ups_registration' ) . "' target='_BLANK'>Click here to Register</a>", 'ups-woocommerce-shipping' ); ?></strong></li>
						<li><?php _e( 'PluginHive, a UPS Ready Business Solutions provider, ensures a seamless transition for all merchants by utilizing UPS Ready Partner Access to create UPS Access Keys while migrating to OAuth and RESTful APIs.', 'ups-woocommerce-shipping' ); ?></li>
						<li></li>

					</ul>

				</div>

			</div>
			<?php
		}

		return ob_get_clean();
	}

	/**
	 * Generate Single Select Country
	 *
	 * @access public
	 * @return void
	 */
	public function generate_single_select_country_html() {

		global $woocommerce;

		$this->origin_country = PH_WC_UPS_Common_Utils::ph_get_origin_country_and_state( $this->settings, 'country' );
		$this->origin_state   = PH_WC_UPS_Common_Utils::ph_get_origin_country_and_state( $this->settings, 'state' );

		ob_start();
		?>

		<tr valign="top" class="ph_ups_general_tab">

			<th scope="row" class="titledesc">
				<label for="origin_country"><?php _e( 'Origin Country', 'ups-woocommerce-shipping' ); ?></label>
			</th>

			<td class="forminp">
				<select name="woocommerce_ups_origin_country_state" id="woocommerce_ups_origin_country_state" style="width: 250px;" data-placeholder="<?php _e( 'Choose a country&hellip;', 'ups-woocommerce-shipping' ); ?>" title="Country" class="chosen_select">
					<?php echo $woocommerce->countries->country_dropdown_options( $this->origin_country, $this->origin_state ? $this->origin_state : '*' ); ?>
				</select>
			</td>

		</tr>

		<?php

		return ob_get_clean();
	}

	/**
	 *
	 * Generate Ship From Country State
	 *
	 * @access public
	 * @return void
	 */
	public function generate_ship_from_country_state_html() {

		global $woocommerce;

		ob_start();
		?>

		<tr valign="top" class="ph_ups_general_tab">

			<th scope="row" class="titledesc">
				<label for="woocommerce_wf_shipping_ups_ship_from_country_state"><?php _e( 'Ship From Country', 'ups-woocommerce-shipping' ); ?></label>
			</th>

			<td class="forminp ph_ups_different_ship_from_address">
				<select name="woocommerce_wf_shipping_ups_ship_from_country_state" id="woocommerce_wf_shipping_ups_ship_from_country_state" style="width: 250px;" data-placeholder="<?php _e( 'Choose a country&hellip;', 'ups-woocommerce-shipping' ); ?>" title="Country" class="chosen_select">
					<?php echo $woocommerce->countries->country_dropdown_options( $this->ship_from_country, $this->ship_from_state ? $this->ship_from_state : '*' ); ?>
				</select>
			</td>

		</tr>

		<?php

		return ob_get_clean();
	}

	/**
	 * Generate Services
	 *
	 * @access public
	 * @return void
	 */
	public function generate_services_html() {

		ob_start();

		include 'admin-views/ph-ups-service-list-html.php';

		return ob_get_clean();
	}

	/**
	 * Generate Box Packing
	 *
	 * @access public
	 * @return void
	 */
	public function generate_box_packing_html() {

		$default_boxes = array();

		if ( ! isset( $this->settings['boxes'] ) || ! is_array( $this->settings['boxes'] ) ) {

			foreach ( $this->packaging as $key => $box ) {

				$default_boxes[ $key ] = array(
					'boxes_name'   => $box['name'],
					'outer_length' => $box['length'],
					'outer_width'  => $box['width'],
					'outer_height' => $box['height'],
					'inner_length' => $box['length'],
					'inner_width'  => $box['width'],
					'inner_height' => $box['height'],
					'box_weight'   => 0,
					'max_weight'   => $box['weight'],
					'max_quantity' => 0,
					'box_enabled'  => true,
				);
			}

			foreach ( $this->simpleRateBoxes as $key => $box ) {

				$default_boxes[ $key ] = array(
					'boxes_name'   => $box['name'],
					'outer_length' => $box['length'],
					'outer_width'  => $box['width'],
					'outer_height' => $box['height'],
					'inner_length' => $box['length'],
					'inner_width'  => $box['width'],
					'inner_height' => $box['height'],
					'box_weight'   => $box['box_weight'],
					'max_weight'   => $box['max_weight'],
					'max_quantity' => 0,
					'box_enabled'  => $box['box_enabled'],
				);
			}
		}

		ob_start();

		include 'admin-views/ph-ups-box-packing-html.php';

		return ob_get_clean();
	}

	/**
	 * Validate Single Select Country
	 *
	 * @access public
	 * @param mixed $key
	 * @return void
	 */
	public function validate_single_select_country_field( $key ) {

		if ( isset( $_POST['woocommerce_ups_origin_country_state'] ) ) {

			return $_POST['woocommerce_ups_origin_country_state'];
		}

		return '';
	}

	/**
	 * Validate Box Packing Field
	 *
	 * @access public
	 * @param mixed $key
	 * @return void
	 */
	public function validate_box_packing_field( $key ) {

		$boxes = array();

		do_action( 'ph_ups_before_validate_box_packing', $_POST );

		if ( isset( $_POST['boxes_outer_length'] ) ) {

			$boxes_name         = $_POST['boxes_name'];
			$boxes_outer_length = $_POST['boxes_outer_length'];
			$boxes_outer_width  = $_POST['boxes_outer_width'];
			$boxes_outer_height = $_POST['boxes_outer_height'];
			$boxes_inner_length = $_POST['boxes_inner_length'];
			$boxes_inner_width  = $_POST['boxes_inner_width'];
			$boxes_inner_height = $_POST['boxes_inner_height'];
			$boxes_box_weight   = $_POST['boxes_box_weight'];
			$boxes_max_weight   = $_POST['boxes_max_weight'];
			$boxes_max_quantity = $_POST['boxes_max_quantity'] ?? array();
			$boxes_enabled      = isset( $_POST['boxes_enabled'] ) ? $_POST['boxes_enabled'] : array();
			$selectedDimUnit    = isset( $_POST['selected_dim_unit'] ) && ! empty( $_POST['selected_dim_unit'] ) ? $_POST['selected_dim_unit'] : $this->units;

			// Update the value as true if simple rate is enabled, to show the simple rate boxes.
			$this->upsSimpleRate = isset( $_POST['woocommerce_wf_shipping_ups_ups_simple_rate'] ) ? true : false;

			if ( ! empty( $boxes_outer_length ) && sizeof( $boxes_outer_length ) ) {

				foreach ( $boxes_outer_length as $key => $value ) {

					if ( $boxes_outer_length[ $key ] && $boxes_outer_width[ $key ] && $boxes_outer_height[ $key ] && $boxes_inner_length[ $key ] && $boxes_inner_width[ $key ] && $boxes_inner_height[ $key ] ) {

						$ph_box_name = str_replace( '\\', '', $boxes_name[ $key ] );

						$boxes[ $key ] = array(
							'boxes_name'   => $ph_box_name,
							'outer_length' => floatval( $boxes_outer_length[ $key ] ),
							'outer_width'  => floatval( $boxes_outer_width[ $key ] ),
							'outer_height' => floatval( $boxes_outer_height[ $key ] ),
							'inner_length' => floatval( $boxes_inner_length[ $key ] ),
							'inner_width'  => floatval( $boxes_inner_width[ $key ] ),
							'inner_height' => floatval( $boxes_inner_height[ $key ] ),
							'box_weight'   => floatval( $boxes_box_weight[ $key ] ),
							'max_weight'   => floatval( $boxes_max_weight[ $key ] ),
							'max_quantity' => ! empty( $boxes_max_quantity ) ? $boxes_max_quantity[ $key ] : 0,
							'box_enabled'  => array_key_exists( $key, $boxes_enabled ) ? true : false,
						);
					}
				}
			}

			// Convert the box weight and dimensions if unit is changed under plugin settings while saving.
			if ( $selectedDimUnit != $this->savedMetrics ) {

				$boxes = PH_WC_UPS_Common_Utils::ph_convert_box_measurement_units( $boxes, $this->units );
			}
		}

		// Delete previous settings for UPS Packaging once updated.
		if ( isset( $this->ups_packaging ) && ! empty( $this->ups_packaging ) ) {

			delete_option( 'ups_packaging' );
		}

		// Sort boxes based on key before saving, to maintain the order while displaying in settings.
		ksort( $boxes );

		do_action( 'ph_ups_after_validate_box_packing', $boxes );

		return $boxes;
	}

	/**
	 * Validate Services
	 *
	 * @access public
	 * @param mixed $key
	 * @return void
	 */
	public function validate_services_field( $key ) {

		$services        = array();
		$posted_services = $_POST['ups_service'];

		foreach ( $posted_services as $code => $settings ) {

			$services[ $code ] = array(
				'name'               => wc_clean( $settings['name'] ),
				'order'              => wc_clean( $settings['order'] ),
				'enabled'            => isset( $settings['enabled'] ) ? true : false,
				'adjustment'         => wc_clean( $settings['adjustment'] ),
				'adjustment_percent' => str_replace( '%', '', wc_clean( $settings['adjustment_percent'] ) ),
			);
		}

		return $services;
	}

	/**
	 * Generate Settings
	 *
	 * @return void
	 */
	public function generate_ph_ups_settings_tabs_html() {

		ob_start();

		include 'admin-views/ph-ups-settings-tabs-html.php';

		return ob_get_clean();
	}

	/**
	 * Generate Help Support Section
	 *
	 * @return void
	 */
	public function generate_help_support_section_html() {

		ob_start();

		include 'admin-views/ph-ups-help-and-support-html.php';

		return ob_get_clean();
	}

	/**
	 * Generate Freight Banner
	 *
	 * @return void
	 */
	public function generate_ph_ups_freight_banner_html() {
		ob_start();
		include_once 'admin-views/ph-ups-freight-banner.php';
		return ob_get_clean();
	}

	/**
	 * Init Form Fields
	 *
	 * @access public
	 * @return void
	 */
	public function init_form_fields() {

		$this->services = PH_WC_UPS_Common_Utils::get_services_based_on_origin( $this->origin_country );

		$this->form_fields = include 'settings/data-ph-ups-settings.php';
	}

	/**
	 * See if method is available based on the package and cart.
	 *
	 * @param array $package Shipping package.
	 * @return bool
	 */
	public function is_available( $package ) {

		if ( 'no' === $this->enabled ) {

			Ph_UPS_Woo_Shipping_Common::phAddDebugLog( 'UPS : Realtime Rates is not enabled', $this->debug );

			return false;
		}

		if ( 'specific' === $this->availability ) {

			if ( is_array( $this->countries ) && ! in_array( $package['destination']['country'], $this->countries ) ) {

				Ph_UPS_Woo_Shipping_Common::phAddDebugLog( 'UPS : Method Availability for Specific Countries - ' . print_r( $this->countries, 1 ), $this->debug );
				Ph_UPS_Woo_Shipping_Common::phAddDebugLog( 'UPS : Checking for Destination - ' . $package['destination']['country'] . ' Rate Calculation Aborted.', $this->debug );

				return false;
			}
		} elseif ( 'excluding' === $this->availability ) {

			if ( is_array( $this->countries ) && ( in_array( $package['destination']['country'], $this->countries ) || ! $package['destination']['country'] ) ) {

				return false;
			}
		}

		$has_met_min_amount = false;

		if ( ! method_exists( WC()->cart, 'get_displayed_subtotal' ) ) { // WC version below 2.6

			$total = WC()->cart->subtotal;
		} else {

			$total = WC()->cart->get_displayed_subtotal();

			if ( version_compare( WC()->version, '4.4', '<' ) ) {

				$tax_display = WC()->cart->tax_display_cart;
			} else {

				$tax_display = WC()->cart->get_tax_price_display_mode();
			}

			if ( 'incl' === $tax_display ) {

				$total = $total - ( WC()->cart->get_cart_discount_total() + WC()->cart->get_cart_discount_tax_total() );
			} else {

				$total = $total - WC()->cart->get_cart_discount_total();
			}
		}

		if ( $total < 0 ) {

			$total = 0;
		}

		if ( $total >= $this->min_amount ) {

			$has_met_min_amount = true;
		}

		$is_available = $has_met_min_amount;

		return apply_filters( 'woocommerce_shipping_' . $this->id . '_is_available', $is_available, $package );
	}

	/**
	 * Calculate Shipping
	 *
	 * @access public
	 * @param mixed $package
	 * @return void
	 */
	public function calculate_shipping( $package = array() ) {

		if ( ! Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() ) {

			Ph_UPS_Woo_Shipping_Common::debug( 'UPS shipping rates cannot be retrieved. Please register your UPS account using UPS OAuth in the Registration section.', $this->debug, $this->silent_debug );

			Ph_UPS_Woo_Shipping_Common::phAddDebugLog( 'UPS shipping rates cannot be retrieved. Please register your UPS account using UPS OAuth in the Registration section.', $this->debug );

			return;
		}

		$package   = apply_filters( 'ph_customize_package_details', $package, 'ups' );
		$all_rates = array();

		$settings_helper = new PH_WC_UPS_Settings_Helper();
		$this->settings  = $settings_helper->settings;

		if ( ! class_exists( 'PH_WC_UPS_Shipping_Controller' ) ) {

			require_once PH_WC_UPS_PLUGIN_DIR_PATH . 'includes/utils/class-ph-ups-shipping-controller.php';
		}

		$shipping_controller = new PH_WC_UPS_Shipping_Controller( $this->settings );

		// Function to get rates for both zones and rest.
		$rates = $shipping_controller->ph_get_shipping_rates( $package, 'UPS' );

		// Incase of no active license or no api details or any other issues, the return value might be empty.
		if ( empty( $rates ) ) {

			return;
		}

		$this->current_package_items_and_quantity = $rates['current_package_items_and_quantity'];
		$this->vendorId                           = $rates['vendor_id'];

		if ( ! empty( $rates ) ) {

			$all_rates = $rates['all_rates'];
		}

		$this->xa_add_rates( $all_rates );
	}

	/**
	 * Add Rates
	 *
	 * Adds UPS shipping rates to the WooCommerce shipping method.
	 *
	 * @param array $rates The array of UPS shipping rates.
	 * @return void
	 */
	function xa_add_rates( $rates ) {

		if ( ! empty( $rates ) ) {

			if ( 'all' == $this->offer_rates ) {

				uasort( $rates, array( 'Ph_UPS_Woo_Shipping_Common', 'sort_rates' ) );

				foreach ( $rates as $key => $rate ) {

					// Adjust the shipping cost to ensure it remains within the defined minimum and maximum limits.
					$rate['cost'] = PH_WC_UPS_Common_Utils::ph_adjust_shipping_cost_within_limits( $rate['cost'], $this->min_shipping_cost, $this->max_shipping_cost );

					$this->add_rate( $rate );
				}
			} else {

				$cheapest_rate = '';

				foreach ( $rates as $key => $rate ) {

					if ( ! $cheapest_rate || ( $cheapest_rate['cost'] > $rate['cost'] && ! empty( $rate['cost'] ) ) ) {

						$cheapest_rate = $rate;
					}
				}

				if ( $cheapest_rate ) {

					// Adjust the shipping cost to ensure it remains within the defined minimum and maximum limits.
					$cheapest_rate['cost'] = PH_WC_UPS_Common_Utils::ph_adjust_shipping_cost_within_limits( $cheapest_rate['cost'], $this->min_shipping_cost, $this->max_shipping_cost );
				}

				// If cheapest only without actual service name i.e Service name has to be overriden with method title.
				if ( ! empty( $this->cheapest_rate_title ) ) {

					$cheapest_rate['label'] = $this->cheapest_rate_title;
				}
				$this->add_rate( $cheapest_rate );
			}

			// Fallback.
		} elseif ( $this->fallback ) {

			$this->add_rate(
				array(
					'id'        => $this->id . '_fallback',
					'label'     => $this->title,
					'cost'      => $this->fallback,
					'sort'      => 0,
					'meta_data' => array(
						'_xa_ups_method' => array(
							'id'           => $this->id . '_fallback',    // Rate id will be in format WF_UPS_ID:service_id ex for ground wf_shipping_ups:03
							'method_title' => $this->title,
							'items'        => isset( $this->current_package_items_and_quantity ) ? $this->current_package_items_and_quantity : array(),
						),
						'VendorId'       => ! empty( $this->vendorId ) ? $this->vendorId : null,
					),
				)
			);

			Ph_UPS_Woo_Shipping_Common::debug( __( 'UPS: Using Fallback Setting.', 'ups-woocommerce-shipping' ), $this->debug, $this->silent_debug );
		}
	}
}
