<?php

use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;

class WF_Shipping_UPS_Admin {

	private static $wc_version;

	private $ups_services = array(
		// Domestic
		'12'   => 'UPS 3 Day Select®',
		'03'   => 'UPS® Ground',
		'02'   => 'UPS 2nd Day Air®',
		'59'   => 'UPS 2nd Day Air A.M.®',
		'01'   => 'UPS Next Day Air®',
		'13'   => 'UPS Next Day Air Saver®',
		'14'   => 'UPS Next Day Air® Early',
		'74'   => 'UPS Express® 12:00',       // Germany Domestic

		// International
		'11'   => 'UPS® Standard',
		'07'   => 'UPS Worldwide Express™',
		'54'   => 'UPS Worldwide Express Plus™',
		'08'   => 'UPS Worldwide Expedited',
		'65'   => 'UPS Worldwide Saver',

		// SurePost
		'92'   => 'UPS SurePost® (USPS) < 1lb',
		'93'   => 'UPS SurePost® (USPS) > 1lb',
		'94'   => 'UPS SurePost® (USPS) BPM',
		'95'   => 'UPS SurePost® (USPS) Media',

		// New Services
		'M2'   => 'UPS First Class Mail',
		'M3'   => 'UPS Priority Mail',
		'M4'   => 'UPS Expedited Mail Innovations ',
		'M5'   => 'UPS Priority Mail Innovations ',
		'M6'   => 'UPS Economy Mail Innovations ',
		'70'   => 'UPS Access Point® Economy ',
		'96'   => 'UPS Worldwide Express Freight',

		'US48' => 'UPS Ground Freight Pricing',
	);

	// Shipments Originating in the European Union
	private $euservices = array(
		'07' => 'UPS Express',
		'08' => 'UPS Expedited',
		'11' => 'UPS® Standard',
		'54' => 'UPS Worldwide Express Plus™',
		'65' => 'UPS Worldwide Saver®',
		'70' => 'UPS Access Point® Economy',
		'74' => 'UPS Express® 12:00',
	);

	private $ukservices = array(
		'07' => 'UPS Express',
		'08' => 'UPS Expedited',
		'11' => 'UPS® Standard',
		'54' => 'UPS Worldwide Express Plus™',
		'65' => 'UPS Worldwide Express Saver®',
		'70' => 'UPS Access Point® Economy',
	);

	// European country
	private $eu_array = array( 'BE', 'BG', 'CZ', 'DK', 'DE', 'EE', 'IE', 'GR', 'ES', 'FR', 'HR', 'IT', 'CY', 'LV', 'LT', 'LU', 'HU', 'MT', 'NL', 'AT', 'PT', 'RO', 'SI', 'SK', 'FI', 'PL', 'SE' );

	private $polandservices = array(
		'07' => 'UPS Express',
		'08' => 'UPS Expedited',
		'11' => 'UPS® Standard',
		'54' => 'UPS Express Plus®',
		'65' => 'UPS Express Saver',
		'82' => 'UPS Today Standard',
		'83' => 'UPS Today Dedicated Courier',
		'85' => 'UPS Today Express',
		'86' => 'UPS Today Express Saver',
		'70' => 'UPS Access Point® Economy',
	);

	private $canadaservices = array(
		'01' => 'UPS Express',
		'02' => 'UPS Expedited',
		'07' => 'UPS Worldwide Express™',
		'08' => 'UPS Worldwide Expedited®',
		'11' => 'UPS® Standard',
		'12' => 'UPS 3 Day Select®',                // For CA and US48
		'13' => 'UPS Express Saver',
		'14' => 'UPS Express Early',
		'54' => 'UPS Worldwide Express Plus™',  // UPS Express Early for CA and US48
		'65' => 'UPS Express Saver',
		'70' => 'UPS Access Point® Economy',
	);

	private $ups_surepost_services = array( 92, 93, 94, 95 );

	private $simpleRateBoxCodes = array(
		'XS'   => 'XS',
		'XS:2' => 'XS',
		'XS:3' => 'XS',
		// Small Boxes
		'S'    => 'S',
		'S:2'  => 'S',
		'S:3'  => 'S',
		// Medium Boxes
		'M'    => 'M',
		'M:2'  => 'M',
		'M:3'  => 'M',
		// Large Boxes
		'L'    => 'L',
		'L:2'  => 'L',
		'L:3'  => 'L',
		// Extra Large Boxes
		'XL'   => 'XL',
		'XL:2' => 'XL',
		'XL:3' => 'XL',
	);

	/**
	 * For Delivery Confirmation below array of countries will be considered as domestic, Confirmed by UPS.
	 * US to US, CA to CA, PR to PR are considered as domestic, all other shipments are international.
	 *
	 * @var array
	 */
	public $dc_domestic_countries = array( 'US', 'CA', 'PR' );

	/**
	 * General Variables
	 */
	public $settings, $ups_settings, $insuredvalue, $isc, $weight_packing_process, $xa_show_all, $address_validation, $residential, $enable_latin_encoding, $debug, $uom, $commercial_invoice, $shippingAddressAsSoldTo, $dangerous_goods_signatoryinfo, $commercial_invoice_shipping, $total_insured_commercial, $terms_of_shipment, $reason_export, $return_reason_export, $endpoint, $usps_endorsement, $min_order_amount_for_insurance, $skip_products, $billing_address_as_shipper, $ship_from_address_different_from_shipper, $enable_density_based_rating, $transportation, $add_product_sku, $include_in_commercial_invoice, $discounted_price, $remove_recipients_phno, $remove_special_char_product, $shipper_release_indicator, $dangerous_goods_manifest, $carbonneutral_indicator, $eu_country_cod_type, $upsSimpleRate, $fixedProductPrice, $wcsups, $invoice_commodity_value;
	/**
	 * Boxes
	 */
	public $boxesFromSettings;
	/**
	 * NAFTA
	 */
	public $nafta_co_form, $nafta_producer_option, $blanket_begin_period, $blanket_end_period;
	/**
	 * Editing for Documents Upload Variable
	 */
	public $editingPost;
	/**
	 * Email Notification
	 */
	public $mail_innovation_type, $email_notification, $email_notification_code;
	/**
	 * Default Recipient phone number
	 */
	public $default_recipient_phone, $default_recipient_phone_num;
	/**
	 * Weight and Dimensions Variables
	 */
	public $box_max_weight, $min_weight_limit, $max_weight_limit, $min_shipping_cost, $max_shipping_cost, $density_length, $density_width, $density_height;
	/**
	 * Tin Variables
	 */
	public $tin_number, $recipients_tin;
	/**
	 * Custom and Duties
	 */
	public $customandduties, $customandduties_pcode, $customandduties_ccode, $customandduties_ac_num;
	/**
	 * Delivery Confirmation
	 */
	public $ph_delivery_confirmation;
	/**
	 * Label Variables
	 */
	public $disble_ups_print_label, $disble_shipment_tracking, $show_label_in_browser, $label_description, $label_custom_description, $order_id_or_number_in_label;
	/**
	 * Accesspoint
	 */
	public $accesspoint_locator, $accesspoint_req_option, $accesspoint_max_limit, $accesspoint_option_code;
	/**
	 * Admin Details
	 */
	public $user_id, $password, $access_key, $shipper_number, $ups_display_name;
	/**
	 * EST Delivery
	 */
	public $show_est_delivery;
	/**
	 * Address Variables
	 */
	public $countries_with_statecodes, $origin_country, $origin_country_state, $origin_state, $origin_addressline, $origin_addressline_2, $origin_city, $origin_postcode;
	/**
	 * Vendor Details
	 */
	public $vendorInfo;
	/**
	 * Unit Variables
	 */
	public $units, $weight_unit, $dim_unit, $density_unit, $invoice_unit_of_measure;
	/**
	 * EEI Variables
	 */
	public $eei_data, $edi_on_label, $eei_shipper_filed_option, $eei_pre_departure_itn_number, $eei_exemption_legend, $eei_mode_of_transport, $eei_parties_to_transaction, $eei_ultimate_consignee_code;
	/**
	 * Auto label generation Variables
	 */
	public $auto_label_generation, $auto_label_services;

	/**
	 *
	 * UPS Shipping Admin class constructor.
	 */
	public function __construct() {

		$this->wf_init();

		if ( is_admin() ) {

			$this->init_bulk_printing();

			// Add UPS Meta Boxes in Edit Order Page
			add_action( 'add_meta_boxes', array( $this, 'ph_add_ups_metabox' ), 15, 2 );

			// Notice System
			add_action( 'admin_notices', array( new wf_admin_notice(), 'throw_notices' ), 15 );
		}

		$is_oauth      = Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer();
		$rest_instance = null;

		if ( ! class_exists( 'PH_Shipping_UPS_Admin_Rest' ) ) {

			include_once 'ups_rest/class-ph-shipping-ups-rest-admin.php';
		}

		if ( $is_oauth ) {

			$rest_instance = new PH_Shipping_UPS_Admin_Rest();
		}

		$this->maybe_register_ups_get_hooks( $is_oauth, $rest_instance );
	}

	/**
	 * Registers action hooks based on UPS-related GET parameters.
	 *
	 * This method maps incoming `$_GET` request parameters to internal UPS
	 * methods or REST handlers and binds them to appropriate WordPress hooks
	 * such as `init` or `admin_init`. OAuth checks are respected for protected routes.
	 *
	 * @param bool                            $is_oauth Whether the current user is authenticated via OAuth.
	 * @param PH_Shipping_UPS_Admin_Rest|null $rest_instance Instance of the UPS REST admin handler class.
	 */
	private function maybe_register_ups_get_hooks( $is_oauth, $rest_instance ) {

		if ( empty( $_GET ) ) {

			return;
		}

		$action_map = array(

			// Generate Packages.
			'phupsgp'                                => array(

				'callback'       => array( $rest_instance, 'ph_ups_generate_packages' ),
				'hook'           => 'init',
				'oauth_required' => true,
			),

			// Rate calculation.
			'wf_ups_generate_packages_rates'         => array(

				'callback'       => array( $this, 'wf_ups_generate_packages_rates' ),
				'hook'           => 'admin_init',
				'oauth_required' => true,
			),

			// Shipment Creation.
			'wf_ups_shipment_confirm'                => array(

				'callback'       => array( $rest_instance, 'wf_ups_shipment_confirm' ),
				'hook'           => 'init',
				'oauth_required' => true,
			),
			'xa_generate_return_label'               => array(

				'callback'       => array( $rest_instance, 'xa_generate_return_label' ),
				'hook'           => 'init',
				'oauth_required' => true,
			),

			// Print-related actions grouped together.
			'wf_ups_print_label'                     => array(

				'callback' => array( $this, 'wf_ups_print_label' ),
				'hook'     => 'init',
			),
			'wf_ups_print_commercial_invoice'        => array(

				'callback' => array( $this, 'wf_ups_print_commercial_invoice' ),
				'hook'     => 'init',
			),
			'wf_ups_print_return_commercial_invoice' => array(

				'callback' => array( $this, 'wf_ups_print_return_commercial_invoice' ),
				'hook'     => 'init',
			),
			'ph_ups_print_control_log_receipt'       => array(

				'callback' => array( $this, 'ph_ups_print_control_log_receipt' ),
				'hook'     => 'admin_init',
			),
			'ph_ups_dgm'                             => array(

				'callback' => array( $this, 'ph_ups_print_dangerous_goods_manifest' ),
				'hook'     => 'admin_init',
			),
			'ph_ups_dangerous_goods_signatoryinfo'   => array(

				'callback' => array( $this, 'ph_ups_print_dangerous_goods_signatoryinfo' ),
				'hook'     => 'admin_init',
			),

			// Void shipment.
			'wf_ups_void_shipment'                   => array(

				'callback'       => array( $rest_instance, 'wf_ups_void_shipment' ),
				'hook'           => 'init',
				'oauth_required' => true,
			),
		);

		foreach ( $action_map as $param => $data ) {

			if ( isset( $_GET[ $param ] ) ) {

				if ( ! empty( $data['oauth_required'] ) && ! $is_oauth ) {

					wp_die(
						wp_kses_post(
							sprintf(
								/* translators: %s: Registration page link */
								__( 'UPS functionality is unavailable. Please register your UPS account using UPS OAuth in the <a href="%s" target="_blank" rel="noopener noreferrer">Registration</a> section to enable these features.', 'ups-woocommerce-shipping' ),
								esc_url( admin_url( 'admin.php?page=ph_ups_registration' ) )
							)
						),
						'',
						array( 'back_link' => true )
					);

				}

				if ( empty( $data['callback'] ) || ! is_callable( $data['callback'] ) ) {

					wp_die(
						esc_html__( 'UPS functionality is temporarily unavailable due to a configuration issue. Please contact support or check the plugin settings.', 'ups-woocommerce-shipping' ),
						'',
						array( 'back_link' => true )
					);
				}

				add_action( $data['hook'], $data['callback'], 15 );

				// Only first matching param is handled.
				break;
			}
		}
	}

	private function wf_init() {

		if ( empty( self::$wc_version ) ) {
			self::$wc_version = WC()->version;
		}

		// Load UPS Settings.
		$this->settings = get_option( 'woocommerce_' . WF_UPS_ID . '_settings', null );

		// Print Label Settings.
		$this->isc                    = isset( $this->settings['international_special_commodities'] ) && ! empty( $this->settings['international_special_commodities'] ) && $this->settings['international_special_commodities'] == 'yes' ? 'yes' : 'no';
		$this->disble_ups_print_label = isset( $this->settings['disble_ups_print_label'] ) ? $this->settings['disble_ups_print_label'] : '';
		$this->show_label_in_browser  = isset( $this->settings['show_label_in_browser'] ) ? $this->settings['show_label_in_browser'] : 'no';

		$this->default_recipient_phone     = ! empty( $this->settings['default_recipient_phone'] ) && 'yes' === $this->settings['default_recipient_phone'] ? true : false;
		$this->default_recipient_phone_num = ! empty( $this->settings['default_recipient_phone_num'] ) ? $this->settings['default_recipient_phone_num'] : '';

		$this->xa_show_all        = isset( $this->settings['xa_show_all'] ) && $this->settings['xa_show_all'] == 'yes' ? true : false;
		$this->address_validation = isset( $this->settings['address_validation'] ) && $this->settings['address_validation'] == 'yes' ? true : false;
		$this->residential        = isset( $this->settings['residential'] ) && $this->settings['residential'] == 'yes' ? true : false;
		$this->units              = isset( $this->settings['units'] ) ? $this->settings['units'] : 'imperial';

		$this->debug        = isset( $this->settings['debug'] ) && $this->settings['debug'] == 'yes' ? true : false;

		if ( $this->units == 'metric' ) {

			$this->weight_unit = 'KGS';
			$this->dim_unit    = 'CM';
		} else {

			$this->weight_unit = 'LBS';
			$this->dim_unit    = 'IN';
		}

		$this->uom = ( $this->units == 'imperial' ) ? 'LB' : 'KG';

		if ( ! class_exists( 'WF_Shipping_UPS' ) ) {

			include_once 'class-wf-shipping-ups.php';
		}

		$this->commercial_invoice            = isset( $this->settings['commercial_invoice'] ) && ! empty( $this->settings['commercial_invoice'] ) && $this->settings['commercial_invoice'] == 'yes' ? true : false;
		$this->shippingAddressAsSoldTo       = isset( $this->settings['sold_to_address'] ) && ! empty( $this->settings['sold_to_address'] ) && $this->settings['sold_to_address'] == 'yes' ? true : false;
		$this->dangerous_goods_signatoryinfo = isset( $this->settings['dangerous_goods_signatoryinfo'] ) && ! empty( $this->settings['dangerous_goods_signatoryinfo'] ) && $this->settings['dangerous_goods_signatoryinfo'] == 'yes' ? true : false;
		$this->nafta_co_form                 = ( isset( $this->settings['nafta_co_form'] ) && ! empty( $this->settings['nafta_co_form'] ) && $this->settings['nafta_co_form'] == 'yes' ) ? true : false;
		$this->eei_data                      = ( isset( $this->settings['eei_data'] ) && ! empty( $this->settings['eei_data'] ) && $this->settings['eei_data'] == 'yes' ) ? true : false;
		$this->edi_on_label                  = ( isset( $this->settings['edi_on_label'] ) && ! empty( $this->settings['edi_on_label'] ) && $this->settings['edi_on_label'] == 'yes' ) ? true : false;
		$this->commercial_invoice_shipping   = ( isset( $this->settings['commercial_invoice_shipping'] ) && ! empty( $this->settings['commercial_invoice_shipping'] ) && $this->settings['commercial_invoice_shipping'] == 'yes' ) ? true : false;
		$this->tin_number                    = isset( $this->settings['tin_number'] ) ? $this->settings['tin_number'] : '';
		$this->recipients_tin                = ( isset( $this->settings['recipients_tin'] ) && ! empty( $this->settings['recipients_tin'] ) && $this->settings['recipients_tin'] == 'yes' ) ? true : false;
		$this->terms_of_shipment             = isset( $this->settings['terms_of_shipment'] ) && ! empty( $this->settings['terms_of_shipment'] ) ? $this->settings['terms_of_shipment'] : '';
		$this->reason_export                 = isset( $this->settings['reason_export'] ) ? $this->settings['reason_export'] : '';
		$this->return_reason_export          = isset( $this->settings['return_reason_export'] ) && ! empty( $this->settings['return_reason_export'] ) ? $this->settings['return_reason_export'] : 'RETURN';
		$this->accesspoint_locator           = ( isset( $this->settings['accesspoint_locator'] ) && $this->settings['accesspoint_locator'] == 'yes' ) ? true : false;
		$this->accesspoint_req_option        = ( isset( $this->settings['accesspoint_req_option'] ) && ! empty( $this->settings['accesspoint_req_option'] ) ) ? $this->settings['accesspoint_req_option'] : '1';
		$this->accesspoint_max_limit         = ( isset( $this->settings['accesspoint_max_limit'] ) && ! empty( $this->settings['accesspoint_max_limit'] ) ) ? $this->settings['accesspoint_max_limit'] : '10';
		$this->accesspoint_option_code       = ( isset( $this->settings['accesspoint_option_code'] ) && ! empty( $this->settings['accesspoint_option_code'] ) ) ? $this->settings['accesspoint_option_code'] : array( '018' );

		// Will Check Old Settings 'include_order_id' and Based on that it will set default for 'order_id_or_number_in_label'
		$default_order_id_or_number        = ( isset( $this->settings['include_order_id'] ) && ! empty( $this->settings['include_order_id'] ) && $this->settings['include_order_id'] == 'yes' ) ? 'include_order_number' : '';
		$this->order_id_or_number_in_label = isset( $this->settings['order_id_or_number_in_label'] ) ? $this->settings['order_id_or_number_in_label'] : $default_order_id_or_number;

		$this->countries_with_statecodes = array( 'US', 'CA', 'IE' );

		$this->set_origin_country_state();

		if ( $this->origin_country == 'PL' ) {

			$this->ups_services = $this->polandservices;
		} elseif ( $this->origin_country == 'CA' ) {

			$this->ups_services = $this->canadaservices;
		} elseif ( $this->origin_country == 'GB' || $this->origin_country == 'UK' ) {

			$this->ups_services = $this->ukservices;
		} elseif ( in_array( $this->origin_country, $this->eu_array ) ) {

			$this->ups_services = $this->euservices;
		}

		$this->mail_innovation_type = ( isset( $this->settings['mail_innovation_type'] ) && ! empty( $this->settings['mail_innovation_type'] ) ) ? $this->settings['mail_innovation_type'] : '66';
		$this->usps_endorsement     = ( isset( $this->settings['usps_endorsement'] ) && ! empty( $this->settings['usps_endorsement'] ) ) ? $this->settings['usps_endorsement'] : '5';

		$this->min_order_amount_for_insurance = ! empty( $this->settings['min_order_amount_for_insurance'] ) ? $this->settings['min_order_amount_for_insurance'] : 0;
		$this->skip_products                  = ! empty( $this->settings['skip_products'] ) ? $this->settings['skip_products'] : array();
		$this->min_weight_limit               = ! empty( $this->settings['min_weight_limit'] ) ? (float) $this->settings['min_weight_limit'] : null;
		$this->max_weight_limit               = ! empty( $this->settings['max_weight_limit'] ) ? (float) $this->settings['max_weight_limit'] : null;
		$this->min_shipping_cost              = ! empty( $this->settings['min_shipping_cost'] ) ? (float) $this->settings['min_shipping_cost'] : null;
		$this->max_shipping_cost              = ! empty( $this->settings['max_shipping_cost'] ) ? (float) $this->settings['max_shipping_cost'] : null;

		$this->billing_address_as_shipper               = ( isset( $this->settings['billing_address_as_shipper'] ) && ! empty( $this->settings['billing_address_as_shipper'] ) && $this->settings['billing_address_as_shipper'] == 'yes' ) ? true : false;
		$this->ship_from_address_different_from_shipper = ! empty( $this->settings['ship_from_address_different_from_shipper'] ) ? $this->settings['ship_from_address_different_from_shipper'] : 'no';

		$this->enable_density_based_rating = ( isset( $this->settings['enable_density_based_rating'] ) && $this->settings['enable_density_based_rating'] == 'yes' ) ? true : false;
		$this->density_length              = ( isset( $this->settings['density_length'] ) && ! empty( $this->settings['density_length'] ) ) ? $this->settings['density_length'] : 0;
		$this->density_width               = ( isset( $this->settings['density_width'] ) && ! empty( $this->settings['density_width'] ) ) ? $this->settings['density_width'] : 0;
		$this->density_height              = ( isset( $this->settings['density_height'] ) && ! empty( $this->settings['density_height'] ) ) ? $this->settings['density_height'] : 0;

		$this->customandduties = ( isset( $this->settings['duties_and_taxes'] ) && ! empty( $this->settings['duties_and_taxes'] ) ) ? $this->settings['duties_and_taxes'] : 'receiver';

		// Third Party Duties And Taxes Payer Options
		$this->customandduties_pcode  = ( $this->customandduties == 'third_party' && isset( $this->settings['shipping_payor_post_code'] ) && ! empty( $this->settings['shipping_payor_post_code'] ) ) ? $this->settings['shipping_payor_post_code'] : '';
		$this->customandduties_ccode  = ( $this->customandduties == 'third_party' && isset( $this->settings['shipping_payor_country_code'] ) && ! empty( $this->settings['shipping_payor_country_code'] ) ) ? $this->settings['shipping_payor_country_code'] : '';
		$this->customandduties_ac_num = ( $this->customandduties == 'third_party' && isset( $this->settings['shipping_payor_acc_no'] ) && ! empty( $this->settings['shipping_payor_acc_no'] ) ) ? $this->settings['shipping_payor_acc_no'] : '';

		$this->transportation = ( isset( $this->settings['transportation'] ) && ! empty( $this->settings['transportation'] ) ) ? $this->settings['transportation'] : 'shipper';

		$this->label_description             = ( isset( $this->settings['label_description'] ) && ! empty( $this->settings['label_description'] ) ) ? $this->settings['label_description'] : 'product_category';
		$this->label_custom_description      = ( isset( $this->settings['label_custom_description'] ) && ! empty( $this->settings['label_custom_description'] ) ) ? $this->settings['label_custom_description'] : '';
		$this->add_product_sku               = ( isset( $this->settings['add_product_sku'] ) && ! empty( $this->settings['add_product_sku'] ) && $this->settings['add_product_sku'] == 'yes' ) ? 'yes' : 'no';
		$this->include_in_commercial_invoice = ( isset( $this->settings['include_in_commercial_invoice'] ) && ! empty( $this->settings['include_in_commercial_invoice'] ) && $this->settings['include_in_commercial_invoice'] == 'yes' ) ? 'yes' : 'no';
		// PDS-125
		$this->discounted_price = ( isset( $this->settings['discounted_price'] ) && ! empty( $this->settings['discounted_price'] ) && $this->settings['discounted_price'] == 'yes' ) ? true : false;

		$this->invoice_commodity_value = isset( $this->settings['invoice_commodity_value'] ) ? $this->settings['invoice_commodity_value'] : ( $this->discounted_price ? 'discount_price' : '' );

		$this->invoice_unit_of_measure = isset( $this->settings['invoice_unit_of_measure'] ) ? $this->settings['invoice_unit_of_measure'] : 'EA';

		$this->remove_recipients_phno      = ( isset( $this->settings['remove_recipients_phno'] ) && ! empty( $this->settings['remove_recipients_phno'] ) && $this->settings['remove_recipients_phno'] == 'yes' ) ? true : false;
		$this->remove_special_char_product = ( isset( $this->settings['remove_special_char_product'] ) && ! empty( $this->settings['remove_special_char_product'] ) && $this->settings['remove_special_char_product'] == 'yes' ) ? true : false;
		$this->shipper_release_indicator   = ( isset( $this->settings['shipper_release_indicator'] ) && ! empty( $this->settings['shipper_release_indicator'] ) && $this->settings['shipper_release_indicator'] == 'yes' ) ? true : false;

		$this->dangerous_goods_manifest = ( isset( $this->settings['dangerous_goods_manifest'] ) && ! empty( $this->settings['dangerous_goods_manifest'] ) && $this->settings['dangerous_goods_manifest'] == 'yes' ) ? true : false;

		$this->carbonneutral_indicator = ( isset( $this->settings['carbonneutral_indicator'] ) && ! empty( $this->settings['carbonneutral_indicator'] ) && $this->settings['carbonneutral_indicator'] == 'yes' ) ? true : false;

		$this->eu_country_cod_type = isset( $this->settings['eu_country_cod_type'] ) && ! empty( $this->settings['eu_country_cod_type'] ) ? $this->settings['eu_country_cod_type'] : 9;
		$this->upsSimpleRate       = isset( $this->settings['ups_simple_rate'] ) && $this->settings['ups_simple_rate'] == 'yes' ? true : false;
		$this->fixedProductPrice   = ( isset( $this->settings['fixed_product_price'] ) && ! empty( $this->settings['fixed_product_price'] ) ) ? $this->settings['fixed_product_price'] : 1;

		$this->wcsups = new WF_Shipping_UPS();

		include_once 'class-wf-shipping-ups-tracking.php';

		// Access Point
		if ( $this->accesspoint_locator && Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() ) {

			if ( ! class_exists( 'PH_Shipping_UPS_Admin_Rest' ) ) {

				include_once 'ups_rest/class-ph-shipping-ups-rest-admin.php';
			}

			$ph_shipping_ups_admin_rest = new PH_Shipping_UPS_Admin_Rest();

			add_action( 'woocommerce_admin_order_data_after_shipping_address', array( $ph_shipping_ups_admin_rest, 'ph_editable_access_point_location' ), 15 );

			add_action( 'woocommerce_process_shop_order_meta', array( $this, 'ph_save_access_point_location' ), 15 );
		}

		$this->user_id              = isset( $this->settings['user_id'] ) ? $this->settings['user_id'] : '';
		$this->password             = isset( $this->settings['password'] ) ? $this->settings['password'] : '';
		$this->access_key           = isset( $this->settings['access_key'] ) ? $this->settings['access_key'] : '';
		$this->shipper_number       = isset( $this->settings['shipper_number'] ) ? $this->settings['shipper_number'] : '';
		$this->ups_display_name     = isset( $this->settings['ups_display_name'] ) ? $this->settings['ups_display_name'] : '';
		$this->origin_addressline   = isset( $this->settings['origin_addressline'] ) ? $this->settings['origin_addressline'] : '';
		$this->origin_addressline_2 = isset( $this->settings['origin_addressline_2'] ) ? $this->settings['origin_addressline_2'] : '';
		$this->origin_city          = isset( $this->settings['origin_city'] ) ? $this->settings['origin_city'] : '';
		$this->origin_postcode      = isset( $this->settings['origin_postcode'] ) ? $this->settings['origin_postcode'] : '';
		$this->show_est_delivery    = ( isset( $this->settings['enable_estimated_delivery'] ) && $this->settings['enable_estimated_delivery'] == 'yes' ) ? true : false;

		$this->ph_delivery_confirmation = isset( $this->settings['ph_delivery_confirmation'] ) && ! empty( $this->settings['ph_delivery_confirmation'] ) ? $this->settings['ph_delivery_confirmation'] : 0;
		$this->vendorInfo               = ( isset( $this->settings['vendor_info'] ) && $this->settings['vendor_info'] == 'yes' ) ? true : false;
	}

	// Needed
	function ph_save_access_point_location( $order_id ) {

		if ( isset( $_POST['shipping_accesspoint'] ) ) {

			$accesspoint_locators = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_accesspoint_location' );
			$accesspoint_id       = $_POST['shipping_accesspoint'];
			$access_locator       = '';

			if ( ! empty( $accesspoint_locators ) ) {

				foreach ( $accesspoint_locators as $locator_id => $locator ) {

					if ( $locator_id == $accesspoint_id ) {

						$access_locator = $accesspoint_locators[ $locator_id ];
						break;
					} else {

						$access_locator = '';
					}
				}
			}

			PH_UPS_WC_Storage_Handler::ph_add_and_save_meta_data( $order_id, '_shipping_accesspoint', $access_locator );
		}
	}

	// Needed
	private function set_origin_country_state( $ups_settings = null ) {
		// Multi-Warehouse Addon Support
		if ( ! empty( $ups_settings ) ) {

			$this->settings = $ups_settings;
		}

		$ups_origin_country_state = isset( $this->settings['origin_country_state'] ) ? $this->settings['origin_country_state'] : '';

		if ( strstr( $ups_origin_country_state, ':' ) ) :
			// WF: Following strict php standards.
			$origin_country_state_array = explode( ':', $ups_origin_country_state );
			$this->origin_country       = current( $origin_country_state_array );
			$origin_country_state_array = explode( ':', $ups_origin_country_state );
			$origin_state               = end( $origin_country_state_array );
		else :
			$this->origin_country = $ups_origin_country_state;
			$origin_state         = '';
		endif;

		$this->origin_state = ( isset( $origin_state ) && ! empty( $origin_state ) ) ? $origin_state : ( isset( $this->settings['origin_custom_state'] ) ? $this->settings['origin_custom_state'] : '' );
	}

	/**
	 * Add UPS Metabox
	 *
	 * @param string $postType
	 * @param object $postObject
	 */
	function ph_add_ups_metabox( $postType, $postObject ) {

		if ( empty( $this->settings ) ) {

			$settings_helper = new PH_WC_UPS_Settings_Helper();
			$this->settings  = $settings_helper->settings;
		}

		if ( 'yes' === $this->disble_ups_print_label ) {
			return;
		}

		$isHPOEnabled = class_exists( '\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController' ) ? wc_get_container()->get( CustomOrdersTableController::class )->custom_orders_table_usage_is_enabled() : false;
		$screenType   = $isHPOEnabled ? wc_get_page_screen_id( 'shop-order' ) : 'shop_order';
		$order_object = ( $postObject instanceof WP_Post ) ? wc_get_order( $postObject->ID ) : ( ( $postObject instanceof WC_Order ) ? $postObject : '' );

		add_meta_box(
			'PH_UPS_Metabox',
			__( 'UPS Shipment Label', 'ups-woocommerce-shipping' ),
			array( $this, 'ph_ups_metabox_content' ),
			$screenType,
			'advanced',
			'high'
		);

		if ( ! empty( $order_object ) ) {

			// Only international shipments support document upload
			$orderDestination = $order_object->get_shipping_country();

			// Document upload is not supported for new registeration
			if ( ( $this->origin_country != $orderDestination ) ) {

				add_meta_box(
					'PH_UPS_DocUpload_Metabox',
					__( 'Upload Additional UPS Documents', 'ups-woocommerce-shipping' ),
					array( $this, 'ph_ups_document_upload_metabox_content' ),
					$screenType,
					'advanced',
					'high'
				);
			}
		}
	}

	/**
	 * UPS Shipment Metabox Content
	 *
	 * @param object $post_or_order_object
	 */
	function ph_ups_metabox_content( $postOrOrderObject ) {

		$order = ( $postOrOrderObject instanceof WP_Post ) ? wc_get_order( $postOrOrderObject->ID ) : $postOrOrderObject;

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		$shipmentId = '';
		$order_id   = $order->get_id();

		if ( ! class_exists( 'PH_Shipping_UPS_Admin_Rest' ) ) {

			include_once 'ups_rest/class-ph-shipping-ups-rest-admin.php';
		}

			$ph_shipping_ups_admin_rest = new PH_Shipping_UPS_Admin_Rest();

			$shipping_service_data = $ph_shipping_ups_admin_rest->wf_get_shipping_service_data( $order );

			$created_shipments_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_rest_created_shipments_details_array' );

		if ( empty( $created_shipments_details_array ) ) {
			$created_shipments_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_created_shipments_details_array' );
		}

		if ( empty( $created_shipments_details_array ) ) {

			$default_service_type = $shipping_service_data['shipping_service'];
			$multiship            = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_multiple_shipping' );

			$download_url    = admin_url( '/?wf_ups_shipment_confirm=' . base64_encode( $shipmentId . '|' . $order_id ) . '&nocache=' . time() );
			$stored_packages = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_wf_ups_stored_packages' );

			$order_insurance_meta = PH_UPS_WC_Storage_Handler::ph_get_meta_data($order_id, '_ph_ups_insurance');
			$insurance_global    = isset($this->settings['insuredvalue']) ? $this->settings['insuredvalue'] : '';

			// This 'insurance_checkbox' will be used to render the checkbox state.
			$insurance_checkbox = (! empty($order_insurance_meta) && 'true' === $order_insurance_meta) ? 'checked' : (('yes' === $insurance_global) ? 'checked' : '');

			// This boolean will control initial display of column/fields.
			$show_insurance_fields = ('checked' === $insurance_checkbox);
			
			if ( empty( $stored_packages ) && ! is_array( $stored_packages ) ) {

				$adminUrl = admin_url( '/?phupsgp=' . base64_encode( $shipmentId . '|' . $order_id ) . '&nocache=' . time() );

				echo '<strong>' . __( 'Step 1: Auto generate packages.', 'ups-woocommerce-shipping' ) . '</strong></br>';
				?>
				<a style="margin: 10px 10px 10px 4px" class="button button-primary tips ups_generate_packages" href="<?php echo $adminUrl; ?>" data-tip="<?php _e( 'Generate Packages', 'ups-woocommerce-shipping' ); ?>">
					<?php _e( 'Generate Packages', 'ups-woocommerce-shipping' ); ?>
				</a>

				<hr style="border-color:#0074a2">

				<?php

			} else {

				echo '<strong>' . __( 'Step 2: Initiate your shipment.', 'ups-woocommerce-shipping' ) . '</strong></br>';

				echo '<ul>';

				echo '<li>';
				echo '<h4>' . __( 'Package(s)', 'ups-woocommerce-shipping' ) . ': </h4>';
				echo '<table style="width:100%;margin: 10px 10px 10px 4px" id="wf_ups_package_list" class="wf-shipment-package-table" style="display: block">';
				echo '<tr>';

				if ( isset( $this->settings['packing_method'] ) && $this->settings['packing_method'] == 'box_packing' ) {

					echo '<th style="width:20%">' . __( 'Box Name', 'ups-woocommerce-shipping' );
				}

				echo '<th>' . __( 'Wt.', 'ups-woocommerce-shipping' ) . '</br>(' . $this->weight_unit . ')</th>';
				echo '<th>' . __( 'L', 'ups-woocommerce-shipping' ) . '</br>(' . $this->dim_unit . ')</th>';
				echo '<th>' . __( 'W', 'ups-woocommerce-shipping' ) . '</br>(' . $this->dim_unit . ')</th>';
				echo '<th>' . __( 'H', 'ups-woocommerce-shipping' ) . '</br>(' . $this->dim_unit . ')</th>';

				// Always render the Insurance header, but hide it if not initially checked
				echo '<th class="ups-insurance-column-header" ' . ( $show_insurance_fields ? '' : 'style="display:none;"' ) . '>';
				echo __( 'Insur.', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( "<div style='text-align :left;'>* Leave as it is if you want to go for Default Insurance Value.<br/>* Enter amount manually if you want to provide Customized Insurance. <br/>* Keep it blank if you do not want insurance for particular package.</div>", 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" />';
				echo '</th>';

				echo '<th>';
				echo __( 'Service', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Contact UPS for more info on this services.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" />';
				echo '</th>';
				echo '<th>';
				_e( 'Remove', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Remove UPS generated packages.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" />';
				echo '</th>';
				echo '<th>&nbsp;</th>';
				echo '</tr>';

				// Get simple rate boxes
				$simple_rate_boxes = PH_WC_UPS_Constants::UPS_SIMPLE_RATE_BOXES_IN_CMS;

				foreach ( $stored_packages as $stored_package_key    => $stored_package ) {

					$dimensions = $this->get_dimension_from_package( $stored_package );

					if ( is_array( $dimensions ) ) {
						?>
						<tr>

							<?php

							if ( isset( $this->settings['packing_method'] ) && $this->settings['packing_method'] == 'box_packing' ) {

								$box_name = isset( $stored_package['Package']['box_name'] ) && ! empty( $stored_package['Package']['box_name'] ) ? $stored_package['Package']['box_name'] : 'Unpacked Product';

								echo '<td><select id="ups_manual_box_name" name="ups_manual_box_name[]" class="select ph-ups-box-name">';

								$selected = false;
								$box_code = isset( $stored_package['Package']['BoxCode'] ) ? $stored_package['Package']['BoxCode'] : '';

								foreach ( $this->settings['boxes'] as $key => $box ) {

									if ( ! $this->upsSimpleRate && array_key_exists( $key, $simple_rate_boxes ) ) {

										continue;
									}

									if ( $box['box_enabled'] ) {

										$selected_attr = ( $box['boxes_name'] === $box_name ) ? ' selected' : '';
										echo '<option value="' . esc_html( $box['boxes_name'] . '_id_' . $key ) . '"' . $selected_attr . '>' . esc_html( $box['boxes_name'] ) . '</option>';
										$selected = $selected || $box['boxes_name'] === $box_name;
									}
								}

								if ( 'Pre-packed Product' == $box_name ) {

									echo '<option value="Pre-packed Product" selected>Pre-packed Product</option>';
								} else {

									echo '<option value="Unpacked Product"' . ( ! $selected ? ' selected' : '' ) . '>Unpacked Product</option>';
								}

								echo '</select>
									  <input type="hidden" id="ups_manual_box_id" name="ups_manual_box_id[]" value="' . $box_code . '"/>
									  </td>';
								?>
								<?php
							}
							?>

							<td><input type="number" id="ups_manual_weight" name="ups_manual_weight[]" style="width:95%" value="<?php echo $dimensions['Weight']; ?>" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );" /></td>
							<td><input type="number" id="ups_manual_length" name="ups_manual_length[]" style="width:95%" value="<?php echo $dimensions['Length']; ?>" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );" /></td>
							<td><input type="number" id="ups_manual_width" name="ups_manual_width[]" style="width:95%" value="<?php echo $dimensions['Width']; ?>" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );" /></td>
							<td><input type="number" id="ups_manual_height" name="ups_manual_height[]" style="width:95%" value="<?php echo $dimensions['Height']; ?>" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );" /></td>

							<!-- Show Insurance field only when Insurance Option is enabled. -->
							<td class="ups-insurance-column-data" <?php echo ( $show_insurance_fields ? '' : 'style="display:none;"' ); ?>>
								<input type="number" id="ups_manual_insurance" name="ups_manual_insurance[]" style="width:95%" value="<?php echo $dimensions['InsuredValue']; ?>" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );" />
							</td>

							<td>
								<select style="width:95%" class="select ups_manual_service" id="ups_manual_service" name="ups_manual_service[]">

									<?php

									if ( $multiship == 'yes' ) {

										$default_service_type = apply_filters( 'ph_ups_modify_shipping_method_service', $default_service_type, $order, $stored_package_key );
									}

									if ( isset( $default_service_type ) && str_ends_with( $default_service_type, '_sat' ) ) {

										$default_service_type = PH_WC_UPS_Common_Utils::ph_remove_sat_del_suffix( $default_service_type );
									}
									
									if ( $this->xa_show_all == true ) {

										foreach ( $this->ups_services as $service_code => $service_name ) {

											echo '<option value="' . $service_code . '" ' . selected( $default_service_type, $service_code ) . ' >' . $service_name . '</option>';
										}
									} elseif ( isset( $this->settings['services'] ) && ! empty( $this->settings['services'] ) ) {

										foreach ( $this->settings['services'] as $service_code => $sdata ) {

											if ( $sdata['enabled'] == 1 ) {

												$service_name = ( isset( $this->ups_services[ $service_code ] ) ) ? $this->ups_services[ $service_code ] : '';

												echo '<option value="' . $service_code . '" ' . selected( $default_service_type, $service_code ) . ' >' . $service_name . '</option>';
											}
										}
									}
									?>

								</select>
							</td>
							<td><a class="wf_ups_package_line_remove" id="<?php echo $stored_package_key; ?>">&#x26D4;</a></td>
							<td>&nbsp;</td>
						</tr>
						<?php
					}
				}

				echo '</table>';
				echo '<div id="ret_s" style="display:none">';
				echo '<h4>' . __( 'Return Package', 'ups-woocommerce-shipping' ) . ': </h4>';
				echo '<table id="rt_wf_ups_package_list" class="wf-shipment-package-table">';
				echo '<tr style="line-height: 2;">';
				echo '<th>' . __( 'Wt.', 'ups-woocommerce-shipping' ) . '</br>(' . $this->weight_unit . ')</th>';
				echo '<th>' . __( 'L', 'ups-woocommerce-shipping' ) . '</br>(' . $this->dim_unit . ')</th>';
				echo '<th>' . __( 'W', 'ups-woocommerce-shipping' ) . '</br>(' . $this->dim_unit . ')</th>';
				echo '<th>' . __( 'H', 'ups-woocommerce-shipping' ) . '</br>(' . $this->dim_unit . ')</th>';
				
				// Return Package Insurance: Also hide/show based on the main checkbox
				echo '<th class="ups-insurance-column-header" ' . ( $show_insurance_fields ? '' : 'style="display:none;"' ) . '>';
				echo __( 'Insur.', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( "<div style='text-align :left;'>* Leave as it is if you want to go for Default Insurance Value.<br/>* Enter amount manually if you want to provide Customized Insurance. <br/>* Keep it blank if you do not want insurance for particular package.</div>", 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" />';
				echo '</th>';

				echo '<th>';
				echo __( 'Service', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Contact UPS for more info on this services.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" />';
				echo '</th>';
				echo '<th>&nbsp;</th>';
				echo '</tr>';

				if ( isset( $dimensions ) && is_array( $dimensions ) ) {

					?>

					<tr>
						<td><input type="number" id="rt_ups_manual_weight" name="rt_ups_manual_weight[]" size="4" value="<?php echo $dimensions['Weight']; ?>" step="any" /></td>
						<td><input type="number" id="rt_ups_manual_length" name="rt_ups_manual_length[]" size="4" value="<?php echo $dimensions['Length']; ?>" step="any" /></td>
						<td><input type="number" id="rt_ups_manual_width" name="rt_ups_manual_width[]" size="4" value="<?php echo $dimensions['Width']; ?>" step="any" /></td>
						<td><input type="number" id="rt_ups_manual_height" name="rt_ups_manual_height[]" size="4" value="<?php echo $dimensions['Height']; ?>" step="any" /></td>
						
						<!-- Return package insurance field: Also hide/show based on the main checkbox -->
						<td class="ups-insurance-column-data" <?php echo ( $show_insurance_fields ? '' : 'style="display:none;"' ); ?>>
							<input type="number" id="rt_ups_manual_insurance" name="rt_ups_manual_insurance[]" size="4" value="<?php echo $dimensions['InsuredValue']; ?>" step="any" />
						</td>
						
						<td>
							<select class="select rt_ups_manual_service" id="rt_ups_manual_service" name="rt_ups_manual_service[]">

								<?php

								foreach ( $this->ups_services as $service_code => $service_name ) {

									echo '<option value="' . $service_code . '" ' . selected( $default_service_type, $service_code ) . ' >' . $service_name . '</option>';
								}

								?>

							</select>
						</td>
						<td>&nbsp;</td>
					</tr>
					<?php
				}

				echo '</table>';
				echo '</div>';

				echo '<a class="button wf-action-button wf-add-button" id="wf_ups_add_package"> ' . __( 'Add Package', 'ups-woocommerce-shipping' ) . '</a>';
				?>
				<a class="button tips ups_generate_packages" href="<?php echo admin_url( '/?phupsgp=' . base64_encode( $shipmentId . '|' . $order_id ) . '&nocache=' . time() ); ?>" data-tip="<?php _e( 'Re-generate all the Packages', 'ups-woocommerce-shipping' ); ?>">
					<?php _e( 'Generate Packages', 'ups-woocommerce-shipping' ); ?>
				</a>
				<?php
				echo '</li>';
				?>

				<script type="text/javascript">
					jQuery(document).ready(function() {

						jQuery(document).on('change', '.ph-ups-box-name', function() {

							let selected_box = jQuery(this).val();;

							let package_name = jQuery(this).val().split('_id_')[0];
							let package_key = jQuery(this).val().split('_id_')[1];

							let all_boxes = <?php echo json_encode( $this->settings['boxes'] ); ?>;

							let chosen_box_details = all_boxes[package_key];

							// Find the parent row of the dropdown
							let parent_row = jQuery(this).closest('tr');

							parent_row.find('#ups_manual_length').val(chosen_box_details['outer_length']).addClass('ph-highlight');
							parent_row.find('#ups_manual_width').val(chosen_box_details['outer_width']).addClass('ph-highlight');
							parent_row.find('#ups_manual_height').val(chosen_box_details['outer_height']).addClass('ph-highlight');

							parent_row.find('#ups_manual_box_id').val(package_key);

							// Remove ph-highlight class after 3 seconds
							setTimeout(function() {
								parent_row.find('#ups_manual_length').removeClass('ph-highlight');
								parent_row.find('#ups_manual_width').removeClass('ph-highlight');
								parent_row.find('#ups_manual_height').removeClass('ph-highlight');
							}, 3000);
						});

						jQuery('#wf_ups_add_package').on("click", function() {

							var new_row = '<tr>';

							<?php if ( isset( $this->settings['packing_method'] ) && 'box_packing' == $this->settings['packing_method'] ) { ?>

								new_row += '<td><select id="ups_manual_box_name" name="ups_manual_box_name[]" class="ph-ups-box-name">';
								<?php

								foreach ( $this->settings['boxes'] as $key => $box ) {

									if ( ! $this->upsSimpleRate && array_key_exists( $key, $simple_rate_boxes ) ) {

										continue;
									}

									if ( $box['box_enabled'] ) {

										?>
										new_row += '<option value="<?php echo $box['boxes_name'] . '_id_' . $key; ?>"><?php echo $box['boxes_name']; ?></option>';
										<?php
									}
								}

								?>

								new_row += '<option value="Unpacked Product" selected>Unpacked Product</option>\
											</select>\
											<input type="hidden" id="ups_manual_box_id" name="ups_manual_box_id[]" value=""/>\
											</td>';

							<?php } ?>
							new_row += '<td><input type="number" id="ups_manual_weight" name="ups_manual_weight[]" style="width:95%" value="0" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );"></td>';
							new_row += '<td><input type="number" id="ups_manual_length" name="ups_manual_length[]" style="width:95%" value="0" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );"></td>';
							new_row += '<td><input type="number" id="ups_manual_width" name="ups_manual_width[]" style="width:95%" value="0" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );"></td>';
							new_row += '<td><input type="number" id="ups_manual_height" name="ups_manual_height[]" style="width:95%" value="0" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );"></td>';

							new_row += '<td class="ups-insurance-column-data"><input type="number" id="ups_manual_insurance" name="ups_manual_insurance[]" style="width:95%" value="0" step="any" min="0" oninput="limitDecimalPlaces(this, 2 );"></td>';


							new_row += '<td>';
							new_row += '<select style="width:95%" class="select ups_manual_service" id="ups_manual_service">';


							<?php

							if ( $this->xa_show_all == true ) {

								foreach ( $this->ups_services as $service_code => $service_name ) {
									?>
									new_row += '<option value="<?php echo $service_code; ?>"><?php echo $service_name; ?></option>';

									<?php
								}
							} else {

								foreach ( $this->settings['services'] as $service_code => $sdata ) {

									if ( $sdata['enabled'] == 1 ) {

										$service_name = isset( $this->ups_services[ $service_code ] ) ? $this->ups_services[ $service_code ] : '';
										?>
										new_row += '<option value="<?php echo $service_code; ?>"><?php echo $service_name; ?></option>';
										<?php
									}
								}
							}
							?>

							new_row += '</select>';
							new_row += '</td>';
							new_row += '<td><a class="wf_ups_package_line_remove">&#x26D4;</a></td>';
							new_row += '</tr>';

							jQuery('#wf_ups_package_list tr:last').after(new_row);
							toggleInsuranceFields(jQuery('#ups_insurance').is(':checked'));

							var rt_new_row = '<tr>';
							rt_new_row += '<td><input type="number" id="rt_ups_manual_weight" name="rt_ups_manual_weight[]" size="2" value="0" step="any"></td>';
							rt_new_row += '<td><input type="number" id="rt_ups_manual_length" name="rt_ups_manual_length[]" size="2" value="0" step="any"></td>';
							rt_new_row += '<td><input type="number" id="rt_ups_manual_width" name="rt_ups_manual_width[]" size="2" value="0" step="any"></td>';
							rt_new_row += '<td><input type="number" id="rt_ups_manual_height" name="rt_ups_manual_height[]" size="2" value="0" step="any"></td>';
							
							rt_new_row += '<td class="ups-insurance-column-data"><input type="number" id="rt_ups_manual_insurance" name="rt_ups_manual_insurance[]" size="2" value="0" step="any"></td>';


							rt_new_row += '<td>';
							rt_new_row += '<select class="select rt_ups_manual_service" id="rt_ups_manual_service">';

							<?php foreach ( $this->ups_services as $service_code => $service_name ) { ?>
								rt_new_row += '<option value="<?php echo $service_code; ?>"><?php echo $service_name; ?></option>';
							<?php } ?>

							rt_new_row += '</select>';
							rt_new_row += '</td>';
							rt_new_row += '<td><a class="wf_ups_package_line_remove">&#x26D4;</a></td>';
							rt_new_row += '</tr>';

							jQuery('#rt_wf_ups_package_list tr:last').after(rt_new_row);
							toggleInsuranceFields(jQuery('#ups_insurance').is(':checked'));

						});

						jQuery(document).on('click', '.wf_ups_package_line_remove', function() {

							jQuery(this).closest('tr').remove();

						});

						// To create Shipment
						jQuery("a.ups_create_shipment").on("click", function(e) {

							// Prevent button click if already clicked
							jQuery(this).css('pointer-events', 'none');
							jQuery(this).attr('disabled', 'disabled');
							jQuery(this).css({
								"color": "#555",
								"background": "#f7f7f7",
								"opacity": ".45",
								"cursor": "not-allowed"
							});

							let shipper_number_exists = false;

							let ph_shipper_number = '';

							// UPS Shipper Number Validation
							if (jQuery('#ph_ups_shipper_number').length != 0 && jQuery("#ph_ups_shipper_number").val() != null) {

								shipper_number_exists = true;
								ph_shipper_number = jQuery("#ph_ups_shipper_number").val();

								if (ph_shipper_number !== '') {

									let validated = true;
									if (ph_shipper_number.length < 6 || (!/\d/.test(ph_shipper_number)) || (!/[a-zA-Z]/.test(ph_shipper_number)) || (/[^0-9a-zA-Z]/.test(ph_shipper_number))) {

										validated = false;
									}

									if (!validated) {

										if (jQuery('#invalid_shipper_no').length === 0) {

											jQuery(".ups_create_shipment").before("<li><label id='invalid_shipper_no' style='color:red'><?php echo __( 'Invalid UPS account number. Please ensure it contains min 6 alphanumeric characters and try again.', 'ups-woocommerce-shipping' ); ?></label></li>");
										}
										e.preventDefault
										return false;
									}
								}
							}

							var manual_box_name_arr = [];

							jQuery('.ph-ups-box-name').each(function() {

								let box_name = jQuery(this).val().split('_id_')[0];

								manual_box_name_arr.push(box_name);
							});

							var manual_box_name = JSON.stringify(manual_box_name_arr);

							var manual_box_id_arr = [];

							var manual_box_id_arr = jQuery("input[id='ups_manual_box_id']").map(function() {
								return jQuery(this).val();
							}).get();

							var manual_box_id = JSON.stringify(manual_box_id_arr);

							var manual_weight_arr = jQuery("input[id='ups_manual_weight']").map(function() {
								return jQuery(this).val();
							}).get();

							var manual_weight = JSON.stringify(manual_weight_arr);

							var manual_height_arr = jQuery("input[id='ups_manual_height']").map(function() {
								return jQuery(this).val();
							}).get();

							var manual_height = JSON.stringify(manual_height_arr);

							var manual_width_arr = jQuery("input[id='ups_manual_width']").map(function() {
								return jQuery(this).val();
							}).get();

							var manual_width = JSON.stringify(manual_width_arr);

							var manual_length_arr = jQuery("input[id='ups_manual_length']").map(function() {
								return jQuery(this).val();
							}).get();

							var manual_length = JSON.stringify(manual_length_arr);

							var manual_insurance_arr = [];
							
							if (jQuery('#ups_insurance').is(':checked')) {
								manual_insurance_arr = jQuery("input[id='ups_manual_insurance']").map(function() {
									return jQuery(this).val();
							}).get();
							} else {
								// If insurance is unchecked, send 0 or null for all packages
								manual_insurance_arr = jQuery("input[id='ups_manual_insurance']").map(function() {
									return 0; // Or null if your backend expects null for no insurance
								}).get();
							}

							var manual_insurance = JSON.stringify(manual_insurance_arr);

							var manual_service_arr = [];

							jQuery('.ups_manual_service').each(function() {

								manual_service_arr.push(jQuery(this).val());
							});

							var manual_service = JSON.stringify(manual_service_arr);

							var rt_manual_weight_arr = jQuery("input[id='rt_ups_manual_weight']").map(function() {
								return jQuery(this).val();
							}).get();

							var rt_manual_weight = JSON.stringify(rt_manual_weight_arr);

							var rt_manual_height_arr = jQuery("input[id='rt_ups_manual_height']").map(function() {
								return jQuery(this).val();
							}).get();

							var rt_manual_height = JSON.stringify(rt_manual_height_arr);

							var rt_manual_width_arr = jQuery("input[id='rt_ups_manual_width']").map(function() {
								return jQuery(this).val();
							}).get();

							var rt_manual_width = JSON.stringify(rt_manual_width_arr);

							var rt_manual_length_arr = jQuery("input[id='rt_ups_manual_length']").map(function() {
								return jQuery(this).val();
							}).get();

							var rt_manual_length = JSON.stringify(rt_manual_length_arr);

							var rt_manual_insurance_arr = [];

							if (jQuery('#ups_insurance').is(':checked')) {
								rt_manual_insurance_arr = jQuery("input[id='rt_ups_manual_insurance']").map(function() {
									return jQuery(this).val();
								}).get();
							} else {
								rt_manual_insurance_arr = jQuery("input[id='rt_ups_manual_insurance']").map(function() {
									return 0;
								}).get();
							}

							var rt_manual_insurance = JSON.stringify(rt_manual_insurance_arr);

							var rt_manual_service_arr = [];

							jQuery('.rt_ups_manual_service').each(function() {

								rt_manual_service_arr.push(jQuery(this).val());
							});

							var rt_manual_service = JSON.stringify(rt_manual_service_arr);

							if (jQuery("#ph_ups_mrn_compliance").val() == null) {
								is_mrn = false;
							} else {
								is_mrn = true;
							}

							if (jQuery("#ph_ups_recipients_tin").val() == null) {
								recipients_tin = false;
							} else {
								recipients_tin = true;
							}

							if (jQuery("#ph_ups_shipto_recipients_tin").val() == null) {
								shipto_recipients_tin = false;
							} else {
								shipto_recipients_tin = true;
							}

							let package_key_arr = [];

							jQuery('.wf_ups_package_line_remove').each(function() {
								package_key_arr.push(this.id);
							});

							let package_key = JSON.stringify(package_key_arr);

							if (jQuery('#ups_return').is(':checked')) {

								var url_location = this.href +
									'&box_id=' + manual_box_id +
									'&box_name=' + manual_box_name +
									'&weight=' + manual_weight +
									'&length=' + manual_length +
									'&width=' + manual_width +
									'&height=' + manual_height +
									'&insurance=' + manual_insurance +
									'&service=' + manual_service +
									'&insr_check=' + jQuery('#ups_insurance').is(':checked') + // Send true/false for checkbox state
									'&cod=' + jQuery('#ups_cod').is(':checked') +
									'&sat_delivery=' + jQuery('#ups_sat_delivery').is(':checked') +
									'&ic=' + jQuery('#ups_import_control').is(':checked') +
									'&rt_weight=' + rt_manual_weight +
									'&rt_length=' + rt_manual_length +
									'&rt_width=' + rt_manual_width +
									'&rt_height=' + rt_manual_height +
									'&rt_insurance=' + rt_manual_insurance +
									'&rt_service=' + rt_manual_service +
									'&is_gfp_shipment=' + jQuery('#ups_gfp_shipment').is(':checked') +
									'&is_return_label=' + jQuery('#ups_return').is(':checked') +
									'&ShipmentTerms=' + jQuery('#terms_of_shipment_service').val() +
									'&dc=' + jQuery('#ups_delivery_signature').val() +
									'&dd=' + jQuery('#ph_ups_direct_delivery').val() +
									'&itn=' + jQuery('#ups_itn_number').val() +
									'&exl=' + jQuery('#ups_exemption_legend').val() +
									'&vci=' + jQuery('#ups_vendor_collect_id').val() +
									'&ct=' + jQuery('#ups_consignee_type').val() +
									'&sfap=' + jQuery('#ph_ship_from_address_preference').val() +
									'&package_key=' + package_key +
									'&uct=' + jQuery('#ph_ultimate_consignee_type').val() +
									'&soldTo=' + jQuery('#ups_shipping_address_as_sold_to').is(':checked');

								if (is_mrn == true) {
									url_location += '&ups_export_compliance=' + jQuery("#ph_ups_mrn_compliance").val();
								}

								if (recipients_tin) {
									url_location += '&ups_recipient_tin=' + jQuery("#ph_ups_recipients_tin").val();
								}

								if (shipto_recipients_tin) {
									url_location += '&ups_shipto_recipient_tin=' + jQuery("#ph_ups_shipto_recipients_tin").val();
								}

								if (shipper_number_exists) {
									url_location += '&ph_ups_shipper_number=' + ph_shipper_number;
								}

								location.href = url_location;

							} else {

								var url_location = this.href +
									'&box_id=' + manual_box_id +
									'&box_name=' + manual_box_name +
									'&weight=' + manual_weight +
									'&length=' + manual_length +
									'&width=' + manual_width +
									'&height=' + manual_height +
									'&insurance=' + manual_insurance +
									'&service=' + manual_service +
									'&insr_check=' + jQuery('#ups_insurance').is(':checked') + // Send true/false for checkbox state.
									'&cod=' + jQuery('#ups_cod').is(':checked') +
									'&sat_delivery=' + jQuery('#ups_sat_delivery').is(':checked') +
									'&ic=' + jQuery('#ups_import_control').is(':checked') +
									'&is_gfp_shipment=' + jQuery('#ups_gfp_shipment').is(':checked') +
									'&is_return_label=' + jQuery('#ups_return').is(':checked') +
									'&ShipmentTerms=' + jQuery('#terms_of_shipment_service').val() +
									'&dc=' + jQuery('#ups_delivery_signature').val() +
									'&dd=' + jQuery('#ph_ups_direct_delivery').val() +
									'&itn=' + jQuery('#ups_itn_number').val() +
									'&exl=' + jQuery('#ups_exemption_legend').val() +
									'&vci=' + jQuery('#ups_vendor_collect_id').val() +
									'&ct=' + jQuery('#ups_consignee_type').val() +
									'&sfap=' + jQuery('#ph_ship_from_address_preference').val() +
									'&package_key=' + package_key +
									'&uct=' + jQuery('#ph_ultimate_consignee_type').val() +
									'&soldTo=' + jQuery('#ups_shipping_address_as_sold_to').is(':checked');

								if (is_mrn == true) {
									url_location += '&ups_export_compliance=' + jQuery("#ph_ups_mrn_compliance").val();
								}

								if (recipients_tin) {
									url_location += '&ups_recipient_tin=' + jQuery("#ph_ups_recipients_tin").val();
								}

								if (shipto_recipients_tin) {
									url_location += '&ups_shipto_recipient_tin=' + jQuery("#ph_ups_shipto_recipients_tin").val();
								}

								if (shipper_number_exists) {
									url_location += '&ph_ups_shipper_number=' + ph_shipper_number;
								}

								location.href = url_location;
							}

							return false;
						});
					});

					/**
					 * Limits the number of decimal places in an input field based on the provided step value.
					 * 
					 * @param {HTMLInputElement} input - The input element to validate.
					 * @param {number} stepVal - The allowed number of decimal places.
					 *
					 * @returns {void}
					 */
					function limitDecimalPlaces(input, stepVal) {

						// Allow empty input since it would be handled when the package is saved.
						if (input.value === '') return;

						// Ensure stepVal is a valid positive integer, defaulting to 1 if invalid.
						var decimalPlaces = Number.isInteger(stepVal) && stepVal >= 0 ? stepVal : 1;

						// Create a regex pattern to allow numbers with up to the specified decimal places.
						const regex = new RegExp(`^\\d*(\\.\\d{0,${decimalPlaces}})?$`);

						// If the input doesn't match the regex, adjust the value to the last valid state.
						if (!regex.test(input.value)) {
							let newValue = input.value;
							while (!regex.test(newValue) && newValue.length > 0) {
								newValue = newValue.slice(0, -1);
							}
							input.value = newValue;
						}
					}
				</script>

				<?php

				// Rates on order page
				$generate_packages_rates = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'wf_ups_generate_packages_rates_response' );

				echo '<li><table id="wf_ups_service_select" class="wf-shipment-package-table" style="margin-bottom: 10px;margin-top: 15px;box-shadow:.5px .5px 5px lightgrey;">';
				echo '<tr>';
				echo '<th>Select Service</th>';
				echo '<th style="text-align:left;padding:5px; font-size:13px;">' . __( 'Service Name', 'ups-woocommerce-shipping' ) . '</th>';
				if ( isset( $this->settings['enable_estimated_delivery'] ) && $this->settings['enable_estimated_delivery'] == 'yes' ) {
					echo '<th style="text-align:left; font-size:13px;">' . __( 'Delivery Time', 'ups-woocommerce-shipping' ) . ' </th>';
				}
				echo '<th style="text-align:left;font-size:13px;">' . __( 'Cost (', 'ups-woocommerce-shipping' ) . get_woocommerce_currency_symbol() . __( ')', 'ups-woocommerce-shipping' ) . ' </th>';
				echo '</tr>';

				echo '<tr>';
				echo "<td style = 'padding-bottom: 10px; padding-left: 15px; '><input name='wf_ups_service_choosing_radio' id='wf_ups_service_choosing_radio' value='wf_ups_individual_service' type='radio' checked='true'></td>";
				echo "<td colspan = '3' style= 'padding-bottom: 10px; text-align:left;'><b>" . __( 'Choose Shipping Methods', 'ups-woocommerce-shipping' ) . '</b> - ' . __( 'Select this option to choose UPS services for each package (Shipping rates will be applied accordingly)', 'ups-woocommerce-shipping' ) . '</td>';
				echo '</tr>';

				if ( ! empty( $generate_packages_rates ) ) {

					$wp_date_format = get_option( 'date_format' );

					foreach ( $generate_packages_rates as $key => $rates ) {

						$ups_service = explode( ':', $rates['id'] );

						echo '<tr style="padding:10px;">';
						echo "<td style = 'padding-left: 15px;'><input name='wf_ups_service_choosing_radio' id='wf_ups_service_choosing_radio' value='" . end( $ups_service ) . "' type='radio' ></td>";
						echo '<td>' . $rates['label'] . '</td>';

						if ( 'yes' === $this->settings['enable_estimated_delivery'] ) {

							$est_delivery_date = isset( $rates['meta_data']['Estimated Delivery'] ) ? date( $wp_date_format, strtotime( $rates['meta_data']['Estimated Delivery'] ) ) : '-';
							echo '<td>' . $est_delivery_date . '</td>';
						}

						echo '<td>' . ( ! empty( $this->settings['conversion_rate'] ) ? $this->settings['conversion_rate'] * $rates['cost'] : $rates['cost'] ) . '</td>';
						echo '</tr>';
					}
				}

				echo '</table></li>';
				// End of Rates on order page
				?>
				<a style="margin: 10px 10px 10px 4px" class="button tips wf_ups_generate_packages_rates button-secondary" href="<?php echo admin_url( '/post.php?wf_ups_generate_packages_rates=' . base64_encode( $order_id ) . '&nocache=' . time() ); ?>" data-tip="<?php _e( 'Calculate the shipping rates for UPS services.', 'ups-woocommerce-shipping' ); ?>">
					<?php _e( 'Calculate Shipping Cost', 'ups-woocommerce-shipping' ); ?>
				</a>
				<?php

				// Start of UPS Special Services Table
				echo '<li><table id="ph_ups_order_edit_page_options" class="ph-ups-order-edit-options-table" style="margin-bottom: 10px;margin-top: 10px;box-shadow:.5px .5px 5px lightgrey;">';
				echo '<tr><th colspan="2"; style="text-align:center;padding:5px; font-size:13px; ">' . __( 'UPS Special Services', 'ups-woocommerce-shipping' ) . '</th>';

				echo '<tr><td>' . __( 'Insurance', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Enable if you want the package content to be insured.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
				echo '<td><label for="ups_insurance"><input type="checkbox" style="" id="ups_insurance" ' . $insurance_checkbox . ' name="ups_insurance" class=""></label></td></tr>';

				// If payment method is COD, check COD by default.
				$order_payment_method = $order->get_payment_method();
				$cod_for_rate         = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_cod' ); // Calculate Rate in Order Edit Page
				$cod_enable           = isset( $this->settings['cod_enable'] ) ? $this->settings['cod_enable'] : ''; // Global Settings
				$cod_checked          = ( $order_payment_method == 'cod' || $cod_for_rate == 'true' || $cod_enable == 'yes' ) ? 'checked' : '';

				echo '<tr><td>' . __( 'Collect On Delivery', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Collect On Delivery would be applicable only for single package which may contain single or multiple product(s).', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
				echo '<td><label for="ups_cod"><input type="checkbox" style="" id="ups_cod" ' . $cod_checked . ' name="ups_cod" class=""></label></td></tr>';

				echo '<tr><td>' . __( 'Include Return Label', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'You can generate the return label only for single package order.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
				echo '<td><label for="ups_return"><input type="checkbox" style="" id="ups_return" name="ups_return" class=""></label></td></tr>';

				$ups_saturday_delivery = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_saturday_delivery' );
				$global_sat_delivery   = isset( $this->settings['saturday_delivery'] ) ? $this->settings['saturday_delivery'] : '';
				$sat_delivery_checked  = ( ! empty( $ups_saturday_delivery ) && $ups_saturday_delivery == 'true' ) ? 'checked' : ( ( $global_sat_delivery == 'yes' ) ? 'checked' : '' );

				echo '<tr><td>' . __( 'Saturday Delivery', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Saturday Delivery from UPS allows you to stretch your business week to Saturday', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
				echo '<td><label for="ups_sat_delivery"><input type="checkbox" style="" id="ups_sat_delivery" ' . $sat_delivery_checked . ' name="ups_sat_delivery" class=""></label></td></tr>';

				$eeiData                  = ( isset( $this->settings['eei_data'] ) && $this->settings['eei_data'] == 'yes' ) ? true : false;
				$eeiShipperFiledOption    = isset( $this->settings['eei_shipper_filed_option'] ) && ! empty( $this->settings['eei_shipper_filed_option'] ) ? $this->settings['eei_shipper_filed_option'] : '';
				$preDepartureItnNumber    = isset( $this->settings['eei_pre_departure_itn_number'] ) && ! empty( $this->settings['eei_pre_departure_itn_number'] ) ? $this->settings['eei_pre_departure_itn_number'] : '';
				$exemptionLegend          = isset( $this->settings['eei_exemption_legend'] ) && ! empty( $this->settings['eei_exemption_legend'] ) ? $this->settings['eei_exemption_legend'] : '';
				$upsUltimateConsigneeType = isset( $this->settings['eei_ultimate_consignee_code'] ) && ! empty( $this->settings['eei_ultimate_consignee_code'] ) ? $this->settings['eei_ultimate_consignee_code'] : '';
				$import_control_settings  = ( isset( $this->settings['import_control_settings'] ) && $this->settings['import_control_settings'] == 'yes' ) ? 'checked' : ''; // UPS Import Control Indicator settings check box

				if ( $order->get_shipping_country() !== $this->origin_country ) {

					$ups_import_control         = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_import_control' );
					$ups_import_control_checked = ( $import_control_settings == 'checked' || $ups_import_control == 'true' ) ? 'checked' : '';

					echo '<tr><td>' . __( 'UPS Import Control', 'ups-woocommerce-shipping' );
					echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'UPS Import Control allows you, as the importer, to initiate UPS shipments from another country and have those shipments delivered to your business or to an alternate location.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
					echo '<td><label for="ups_import_control"><input type="checkbox" style="" id="ups_import_control" name="ups_import_control" class="ups_import_control" ' . $ups_import_control_checked . '></label></td></tr>';

					$shippingAddressAsSoldTo        = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_shipping_address_as_sold_to' );
					$shippingAddressAsSoldToEnabled = ( $this->shippingAddressAsSoldTo || $shippingAddressAsSoldTo ) ? 'checked' : '';

					_e( '<tr><td>Consider Shipping Address as Sold to Address', 'ups-woocommerce-shipping' );
					echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Enabling this option will consider Shipping Address as the Sold To Address for International Shipments.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
					echo '<td><label for="ups_shipping_address_as_sold_to"><input type="checkbox" ' . $shippingAddressAsSoldToEnabled . ' id="ups_shipping_address_as_sold_to" name="ups_shipping_address_as_sold_to" class=""></label></td></tr>';
					echo '</tr>';

					if ( $this->vendorInfo ) {

						$VCIDNumber    = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_vcid_number' );
						$VCIDConsignee = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_vcid_consignee' );

						// Update defult vendor_collect_id_number
						if ( empty( $VCIDNumber ) && isset( $this->settings['vendor_collect_id_number'] ) ) {

							$VCIDNumber = $this->settings['vendor_collect_id_number'];
						}

						// Update defult default_consignee_type
						if ( empty( $VCIDConsignee ) && isset( $this->settings['default_consignee_type'] ) ) {

							$VCIDConsignee = $this->settings['default_consignee_type'];
						}

						echo '<tr><td>' . __( 'Vendor Collect ID Number', 'ups-woocommerce-shipping' );
						echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Presence of this number as part of the shipment information implies the shipper has collected/paid the required VAT tax (outside of UPS/UPS systems). Vendor Colect ID Number will be printed on commercial invoice if present.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
						echo '<td><label for="ups_vendor_collect_id"><input type="text" style="width:50%" id="ups_vendor_collect_id" name="ups_vendor_collect_id" value="' . $VCIDNumber . '"></label></td></tr>';

						$consigneeType = array(
							''   => __( 'None', 'ups-woocommerce-shipping' ),
							'01' => __( 'Business', 'ups-woocommerce-shipping' ),
							'02' => __( 'Consumer', 'ups-woocommerce-shipping' ),
							'NA' => __( 'Not Applicable', 'ups-woocommerce-shipping' ),
						);
						_e( '<tr><td>Consignee Type', 'ups-woocommerce-shipping' );
						echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Select Consignee Type', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
						echo '<td><select id="ups_consignee_type" style= "width:50%">';

						foreach ( $consigneeType as $key => $value ) {

							if ( $key == $VCIDConsignee ) {

								echo "<option value='" . $key . "' selected>" . $value . '</option>';
							} else {
								echo "<option value='" . $key . "'>" . $value . '</option>';
							}
						}
						echo '</select></td></tr>';
					}

					if ( $eeiData && ( ! empty( $eeiShipperFiledOption ) && $eeiShipperFiledOption == 'A' ) ) {

						$upsItnNumber          = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_eei_itn_number' );
						$preDepartureItnNumber = isset( $upsItnNumber ) && ! empty( $upsItnNumber ) ? $upsItnNumber : $preDepartureItnNumber;

						echo '<tr><td>' . __( 'Pre Departure ITN Number', 'ups-woocommerce-shipping' );
						echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Input for Shipper Filed option A. The format is available from AESDirect website', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
						echo '<td><label for="ups_itn_number"><input type="text" style="width:50%" id="ups_itn_number" name="ups_itn_number" value="' . $preDepartureItnNumber . '"></label></td></tr>';
					} elseif ( $eeiData && ( ! empty( $eeiShipperFiledOption ) && $eeiShipperFiledOption == 'B' ) ) {

						$upsexemptionLegend = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_eei_exemption_legend' );
						$exemptionLegend    = isset( $upsexemptionLegend ) && ! empty( $upsexemptionLegend ) ? $upsexemptionLegend : $exemptionLegend;

						echo '<tr><td>' . __( 'Exemption Legend', 'ups-woocommerce-shipping' );
						echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Input for Shipper Filed option B', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
						echo '<td><label for="ups_exemption_legend"><input type="text" style="width:50%" id="ups_exemption_legend" name="ups_exemption_legend" value="' . $exemptionLegend . '" ></label></td></tr>';
					}

					// PDS-130
					$shipment_terms = array(
						''    => __( 'NONE', 'ups-woocommerce-shipping' ),
						'CFR' => __( 'Cost and Freight', 'ups-woocommerce-shipping' ),
						'CIF' => __( 'Cost Insurance and Freight', 'ups-woocommerce-shipping' ),
						'CIP' => __( 'Carriage and Insurance Paid', 'ups-woocommerce-shipping' ),
						'CPT' => __( 'Carriage Paid To', 'ups-woocommerce-shipping' ),
						'DAF' => __( 'Delivered at Frontier', 'ups-woocommerce-shipping' ),
						'DDP' => __( 'Delivery Duty Paid', 'ups-woocommerce-shipping' ),
						'DDU' => __( 'Delivery Duty Unpaid', 'ups-woocommerce-shipping' ),
						'DEQ' => __( 'Delivered Ex Quay', 'ups-woocommerce-shipping' ),
						'DES' => __( 'Delivered Ex Ship', 'ups-woocommerce-shipping' ),
						'EXW' => __( 'Ex Works', 'ups-woocommerce-shipping' ),
						'FAS' => __( 'Free Alongside Ship', 'ups-woocommerce-shipping' ),
						'FCA' => __( 'Free Carrier', 'ups-woocommerce-shipping' ),
						'FOB' => __( 'Free On Board', 'ups-woocommerce-shipping' ),
					);
					_e( '<tr><td>Terms of shipment ', 'ups-woocommerce-shipping' );
					echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Indicates the rights to the seller from the buyer, internationally.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
					echo '<td><select id="terms_of_shipment_service" style= "width:50%">';

					foreach ( $shipment_terms as $key => $value ) {

						if ( $key == $this->terms_of_shipment ) {
							echo "<option value='" . $key . "' selected>" . $value . '</option>';
						} else {
							echo "<option value='" . $key . "'>" . $value . '</option>';
						}
					}

					echo '</select></td></tr>';

					// EEI Data - Ultimate Consignee Type
					if ( $eeiData ) {

						$selectedUltimateConsigneeType = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_eei_ultimate_consignee_type' );
						$upsUltimateConsigneeType      = ( isset( $selectedUltimateConsigneeType ) && ! empty( $selectedUltimateConsigneeType ) ) ? $selectedUltimateConsigneeType : $upsUltimateConsigneeType;

						$ultimateConsigneeTypes = array(
							'none' => __( 'None', 'ups-woocommerce-shipping' ),
							'D'    => __( 'Direct Consumer', 'ups-woocommerce-shipping' ),
							'G'    => __( 'Government Entity', 'ups-woocommerce-shipping' ),
							'R'    => __( 'Reseller', 'ups-woocommerce-shipping' ),
							'O'    => __( 'Other/Unknown', 'ups-woocommerce-shipping' ),
						);

						_e( '<tr><td>Ultimate Consignee Type', 'ups-woocommerce-shipping' );
						echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'An ultimate consignee is the party who will be the final recipient of a shipment.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
						echo '<td><select id="ph_ultimate_consignee_type" style= "width:50%">';

						foreach ( $ultimateConsigneeTypes as $key => $type ) {

							if ( $key == $upsUltimateConsigneeType ) {

								echo "<option value='" . $key . "' selected>" . $type . '</option>';
							} else {

								echo "<option value='" . $key . "'>" . $type . '</option>';
							}
						}

						echo '</select></td></tr>';
					}
				}

				$signature_option = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_delivery_signature' );
				$signature_option = isset( $signature_option ) ? $signature_option : 4;

				// Default it will select - consider as empty
				$delivery_confirmations = array(
					4 => __( '- As per plugin settings -', 'ups-woocommerce-shipping' ),
					0 => __( 'Confirmation Not Required', 'ups-woocommerce-shipping' ),
					1 => __( 'Confirmation Required', 'ups-woocommerce-shipping' ),
					2 => __( 'Confirmation With Signature', 'ups-woocommerce-shipping' ),
					3 => __( 'Confirmation With Adult Signature', 'ups-woocommerce-shipping' ),
				);

				_e( '<tr><td>Delivery Confirmation', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Appropriate signature option for your shipping service.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
				echo '<td><select id="ups_delivery_signature" style= "width:50%">';

				foreach ( $delivery_confirmations as $key => $value ) {

					if ( $key == $signature_option ) {

						echo "<option value='" . $key . "' selected>" . $value . '</option>';
					} else {

						echo "<option value='" . $key . "'>" . $value . '</option>';
					}
				}

				echo '</select></td></tr>';

				$selected_option = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_direct_delivery' );
				$selected_option = isset( $selected_option ) && ! empty( $selected_option ) ? $selected_option : '';

				$direct_delivery = array(
					''    => __( '- As per plugin settings -', 'ups-woocommerce-shipping' ),
					'yes' => __( 'Enable', 'ups-woocommerce-shipping' ),
					'no'  => __( 'Disable', 'ups-woocommerce-shipping' ),
				);

				_e( '<tr><td>Direct Delivery ', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Click here to select Direct Delivery Only', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
				echo '<td><select id="ph_ups_direct_delivery" style= "width:50%">';

				foreach ( $direct_delivery as $key => $value ) {

					if ( $key == $selected_option ) {
						echo "<option value='" . $key . "' selected>" . $value . '</option>';
					} else {
						echo "<option value='" . $key . "'>" . $value . '</option>';
					}
				}

				echo '</select></td></tr>';

				$ship_from_address          = isset( $this->settings['ship_from_address'] ) ? $this->settings['ship_from_address'] : 'origin_address';
				$selected_ship_from_address = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_shipfrom_address_preference' );
				$selected_ship_from_address = ! empty( $selected_ship_from_address ) ? $selected_ship_from_address : $ship_from_address;

				// Multi-vendor addon support - Vendor address
				if ( $ship_from_address == 'vendor_address' ) {

					$address_preference = apply_filters(
						'wf_filter_label_ship_from_address_options',
						array(
							'origin_address'  => __( 'Origin Address', 'ups-woocommerce-shipping' ),
							'billing_address' => __( 'Shipping Address', 'ups-woocommerce-shipping' ),
						)
					);
				} else {

					$address_preference = array(
						'origin_address'  => __( 'Origin Address', 'ups-woocommerce-shipping' ),
						'billing_address' => __( 'Shipping Address', 'ups-woocommerce-shipping' ),
					);
				}

				_e( '<tr><td>Ship From Address Preference', 'ups-woocommerce-shipping' );
				echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Change the preference of Ship From Address printed on the label. You can make use of Billing Address from Order admin page, if you ship from a different location other than shipment origin address.', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
				echo '<td><select id="ph_ship_from_address_preference" style= "width:50%">';

				foreach ( $address_preference as $key => $value ) {

					if ( $key == $selected_ship_from_address ) {
						echo "<option value='" . $key . "' selected>" . $value . '</option>';
					} else {
						echo "<option value='" . $key . "'>" . $value . '</option>';
					}
				}

				echo '</select></td></tr>';
				// If $ship_from_address is vendor_address, then hiding the ship from address select option from order edit page.
				?>
				<script type="text/javascript">
					jQuery(document).ready(function() {

						let ship_from_address = '<?php echo $ship_from_address; ?>';

						if (ship_from_address == 'vendor_address') {

							jQuery("#ph_ship_from_address_preference").closest('tr').hide();
						} else {
							jQuery("#ph_ship_from_address_preference").closest('tr').show();
						}

					});
				</script>
				<?php
				// End of UPS Special Services Table

				$items_cost                           = $order->get_subtotal();
				$order_currency                       = $order->get_currency();
				$mrn_post_currency                    = 'EUR';
				$woocommerce_currency_conversion_rate = get_option( 'woocommerce_multicurrency_rates' );

				if ( $order_currency != $mrn_post_currency && ! empty( $woocommerce_currency_conversion_rate ) ) {

					$mrn_currency_rate   = $woocommerce_currency_conversion_rate[ $mrn_post_currency ];
					$order_currency_rate = $woocommerce_currency_conversion_rate[ $order_currency ];

					$conversion_rate = $mrn_currency_rate / $order_currency_rate;
					$items_cost     *= $conversion_rate;
				}

				$shipping_country = $order->get_shipping_country();

				if ( $this->origin_country == 'DE' && ( $items_cost > 1000 && $shipping_country != 'DE' ) ) {

					$mrn_export_compliance = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_export_compliance' );

					?>
					<li>
						<tr>
							<?php
							echo '<td>' . __( 'Movement Reference Number (MRN)', 'ups-woocommerce-shipping' );
							echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Export Declaration (MRN) for international shippment from Germany', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
							?>
							<td><input type="text" name="ph_ups_mrn_compliance" value="<?php echo $mrn_export_compliance; ?>" id="ph_ups_mrn_compliance"></td>
						<tr>
					</li>
					<?php
				}

				if ( $this->recipients_tin && $this->commercial_invoice ) {

					$recipients_tin        = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_shipping_tax_id_num' );
					$shipto_recipients_tin = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_ship_to_tax_id_num' );

					// Meta to check Ship To Different Address is enabled or not
					$ship_to_different_address = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_ship_to_different_address' );

					// If Ship To Different Address is not enabled, assign Billing TIN to Shipping TIN
					if ( ! $ship_to_different_address ) {
						$shipto_recipients_tin = $recipients_tin;
					}

					?>
					<li>
						<?php
						echo '<tr><td>' . __( 'Recipients Billing Tax Identification Number', 'ups-woocommerce-shipping' );
						echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Recipients Billing Tax Identification Number will be added to International forms', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
						?>
						<td><input type="text" name="ph_ups_recipients_tin" value="<?php echo $recipients_tin; ?>" id="ph_ups_recipients_tin"></td>
						</tr>
					</li>

					<li>
						<?php
						echo '<tr><td>' . __( 'Recipients Shipping Tax Identification Number', 'ups-woocommerce-shipping' );
						echo '<img class="help_tip" style="float:none;" data-tip="' . __( 'Recipients Shipping Tax Identification Number will be added to International forms', 'ups-woocommerce-shipping' ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" height="16" width="16" /></td>';
						?>
						<td><input type="text" name="ph_ups_shipto_recipients_tin" value="<?php echo $shipto_recipients_tin; ?>" id="ph_ups_shipto_recipients_tin"></td>
						</tr>
					</li>
					<?php
				}

				// To add custom field
				do_action( 'ph_ups_add_fields_in_edit_order_page', $order_id );

				echo '</table></li>';

				?>
				<li>
					<a style="margin: 10px 10px 10px 4px" class="button button-primary tips ups_create_shipment" href="<?php echo $download_url; ?>" data-tip="<?php _e( 'Confirm Shipment', 'ups-woocommerce-shipping' ); ?>"><?php _e( 'Confirm Shipment', 'ups-woocommerce-shipping' ); ?></a>
					<hr style="border-color:#0074a2">
				</li>
				<?php

				do_action( 'ph_add_additional_options_after_confirm_shipment', $order_id );
			}

			?>

			<script type="text/javascript">
				// To get rates on order page
				jQuery("a.wf_ups_generate_packages_rates").one("click", function() {

					let package_key_arr = [];
					let manual_box_name_arr = [];

					jQuery('.ph-ups-box-name').each(function() {

						let box_name = jQuery(this).val().split('_id_')[0];

						manual_box_name_arr.push(box_name);
					});

					let manual_box_name = JSON.stringify(manual_box_name_arr);

					let manual_box_id_arr = jQuery("input[id='ups_manual_box_id']").map(function() {
						return jQuery(this).val();
					}).get();

					let manual_box_id = JSON.stringify(manual_box_id_arr);

					jQuery('.wf_ups_package_line_remove').each(function() {
						package_key_arr.push(this.id);
					});

					let package_key = JSON.stringify(package_key_arr);

					jQuery(this).click(function() {
						return false;
					});
					var manual_weight_arr = jQuery("input[id='ups_manual_weight']").map(function() {
						return jQuery(this).val();
					}).get();
					var manual_height_arr = jQuery("input[id='ups_manual_height']").map(function() {
						return jQuery(this).val();
					}).get();
					var manual_width_arr = jQuery("input[id='ups_manual_width']").map(function() {
						return jQuery(this).val();
					}).get();
					var manual_length_arr = jQuery("input[id='ups_manual_length']").map(function() {
						return jQuery(this).val();
					}).get();
					
					var manual_insurance_arr = [];
					if (jQuery('#ups_insurance').is(':checked')) {
						manual_insurance_arr = jQuery("input[id='ups_manual_insurance']").map(function() {
							return jQuery(this).val();
						}).get();
					} else {
						// Send 0 or null for all if checkbox is unchecked
						manual_insurance_arr = jQuery("input[id='ups_manual_insurance']").map(function() {
							return 0; // Or null
						}).get();
					}

					var manual_signature = jQuery('#ups_delivery_signature').val();
					var manual_saturday_delivery = jQuery('#ups_sat_delivery').is(':checked');
					manual_saturday_delivery = JSON.stringify(manual_saturday_delivery);
					var manual_import_control = jQuery('#ups_import_control').is(':checked');
					manual_import_control = JSON.stringify(manual_import_control);
					var manual_insurance_checked = jQuery('#ups_insurance').is(':checked'); // Still send the checkbox state
					manual_insurance_checked = JSON.stringify(manual_insurance_checked);
					var manual_ups_cod = jQuery('#ups_cod').is(':checked');
					manual_ups_cod = JSON.stringify(manual_ups_cod);
					var manual_direct_delivery = jQuery('#ph_ups_direct_delivery').val();

					location.href = this.href +
						'&box_id=' + manual_box_id +
						'&box_name=' + manual_box_name +
						'&weight=' + manual_weight_arr +
						'&length=' + manual_length_arr +
						'&width=' + manual_width_arr +
						'&height=' + manual_height_arr +
						'&insurance=' + manual_insurance_arr +
						'&package_key=' + package_key +
						'&sig=' + manual_signature +
						'&satdel=' + manual_saturday_delivery +
						'&impc=' + manual_import_control +
						'&insur=' + manual_insurance_checked +
						'&cod=' + manual_ups_cod +
						'&ddo=' + manual_direct_delivery;
					return false;
				});
				//End of jQuery for getting the rates

				//For sitching between the services of get rates and services after every generated packages
				jQuery(document).ready(function() {

					jQuery(document).on("change", "#wf_ups_service_choosing_radio", function() {
    			    	var selectedValue = jQuery("#wf_ups_service_choosing_radio:checked").val();
									
    			    	if (selectedValue == 'wf_ups_individual_service') {
    			    	    jQuery(".ups_manual_service").prop("disabled", false);
    			    	} else {
    			    	    // Strip '_sat' suffix if present for package dropdown matching
    			    	    var packageServiceValue = selectedValue;
    			    	    if (selectedValue && selectedValue.endsWith && selectedValue.endsWith('_sat')) {
    			    	        packageServiceValue = selectedValue.replace('_sat', '');
    			    	    }
						
    			    	    jQuery(".ups_manual_service").val(packageServiceValue).change();
    			    	    jQuery(".ups_manual_service").prop("disabled", true);
    			    	}
    				});
				});
				//End For sitching between the services of get rates and services after every generated packages
			</script>
			<?php

		} else {

			$ups_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_rest_label_details_array' );

			$ups_commercial_invoice_details        = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_commercial_invoice_details' );
			$ups_return_commercial_invoice_details = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_return_commercial_invoice_details' );
			$ups_control_log_receipt               = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_control_log_receipt' );
			$ups_dangerous_goods_manifest_required = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_dangerous_goods_manifest_required' );
			$ups_dangerous_goods_manifest_data     = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_dangerous_goods_manifest_data' );
			$ups_dangerous_goods_image             = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_dangerous_goods_image' );

			if ( empty( $ups_dangerous_goods_image ) ) {

				// Get from custom meta table
				$ups_dangerous_goods_image = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_dangerous_goods_image' );
			}

			$custom_tracking     = isset( $this->settings['custom_tracking'] ) && $this->settings['custom_tracking'] == 'yes' ? true : false;
			$custom_tracking_url = isset( $this->settings['custom_tracking_url'] ) ? $this->settings['custom_tracking_url'] : '';

			if ( empty( $ups_label_details_array ) ) {
				$ups_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_label_details_array' );
			}

			if ( ! empty( $ups_label_details_array ) && is_array( $ups_label_details_array ) ) {

				// For displaying the products name with label on order page
				$packages = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_wf_ups_stored_packages' );

				foreach ( $created_shipments_details_array as $shipmentId => $created_shipments_details ) {

					if ( 'yes' == $this->show_label_in_browser ) {

						$target_val = '_blank';
					} else {

						$target_val = '_self';
					}

					// Multiple labels for each package.
					$index = 0;

					if ( ! empty( $ups_label_details_array[ $shipmentId ] ) ) {

						// Index of Label in the case of making Forward label by using PH UPS Return Label Type addon
						$index_for_label_count = 0;

						foreach ( $ups_label_details_array[ $shipmentId ] as $ups_label_details ) {

							$label_extn_code = $ups_label_details['Code'];
							$tracking_number = isset( $ups_label_details['TrackingNumber'] ) ? $ups_label_details['TrackingNumber'] : '';
							$download_url    = admin_url( '/?wf_ups_print_label=' . base64_encode( $shipmentId . '|' . $order_id . '|' . $label_extn_code . '|' . $index . '|' . $tracking_number ) . '&nocache=' . time() );
							$post_fix_label  = '';

							if ( count( $ups_label_details_array ) > 1 ) {

								$post_fix_label = '#' . ( $index + 1 );
							}

							// Stored packages will be in array format only if it has not been messed manually
							if ( is_array( $packages ) ) {
								?>

								<table class="xa_ups_shipment_box_table" style="border:1px solid lightgray;margin: 5px;margin-top: 5px;box-shadow:.5px .5px 5px lightgrey; width:100%;">
									<caption style="font-size: 16px; color:#E74C3C;">Package Details</caption>
									<tr>
										<?php if ( isset( $this->settings['packing_method'] ) && 'box_packing' == $this->settings['packing_method'] ) { ?>

											<th style="font-size:16px;"><?php echo __( 'Box Name', 'ups-woocommerce-shipping' ); ?></th>

										<?php } ?>
										<th style="font-size:16px;"><?php echo __( 'Weight', 'ups-woocommerce-shipping' ); ?></th>
										<th style="font-size:16px;"><?php echo __( 'Length', 'ups-woocommerce-shipping' ); ?></th>
										<th style="font-size:16px;"><?php echo __( 'Width', 'ups-woocommerce-shipping' ); ?></th>
										<th style="font-size:16px;"><?php echo __( 'Height', 'ups-woocommerce-shipping' ); ?></th>
										<th style="font-size:16px;"><?php echo __( 'Products ( Name x Quantity )', 'ups-woocommerce-shipping' ); ?></th>
									</tr>

								<?php
								$package        = array_shift( $packages );
								$package_weight = $package['Package']['PackageWeight']['Weight'] . ' ' . ( isset( $package['Package']['PackageWeight']['UnitOfMeasurement'] ) ? $package['Package']['PackageWeight']['UnitOfMeasurement']['Code'] : $this->weight_unit );
								$package_length = ( ! empty( $package['Package']['Dimensions'] ) && ! empty( $package['Package']['Dimensions']['Length'] ) ) ? ( $package['Package']['Dimensions']['Length'] . ' ' . ( isset( $package['Package']['Dimensions']['UnitOfMeasurement'] ) ? $package['Package']['Dimensions']['UnitOfMeasurement']['Code'] : '' ) ) : 0;
								$package_width  = ( ! empty( $package['Package']['Dimensions'] ) && ! empty( $package['Package']['Dimensions']['Width'] ) ) ? ( $package['Package']['Dimensions']['Width'] . ' ' . ( isset( $package['Package']['Dimensions']['UnitOfMeasurement'] ) ? $package['Package']['Dimensions']['UnitOfMeasurement']['Code'] : '' ) ) : 0;
								$package_height = ( ! empty( $package['Package']['Dimensions'] ) && ! empty( $package['Package']['Dimensions']['Height'] ) ) ? ( $package['Package']['Dimensions']['Height'] . ' ' . ( isset( $package['Package']['Dimensions']['UnitOfMeasurement'] ) ? $package['Package']['Dimensions']['UnitOfMeasurement']['Code'] : '' ) ) : 0;
								$ph_box_name    = ! empty( $package['Package']['box_name'] ) ? $package['Package']['box_name'] : 'Unpacked Product';

								if ( isset( $this->settings['packing_method'] ) && 'box_packing' == $this->settings['packing_method'] ) {

									echo "<td style='text-align:center; padding: 5px; font-size:16px;'>" . $ph_box_name . '</td>';
								}

								echo "<td style='text-align:center; padding: 5px; font-size:16px;'>" . $package_weight . '</td>';
								echo "<td style='text-align:center; padding: 5px; font-size:16px;'>" . $package_length . '</td>';
								echo "<td style='text-align:center; padding: 5px; font-size:16px;'>" . $package_width . '</td>';
								echo "<td style='text-align:center; padding: 5px; font-size:16px;'>" . $package_height . '</td>';

								$first_item_in_package = ( isset( $package['Package']['items'] ) && is_array( $package['Package']['items'] ) ) ? current( $package['Package']['items'] ) : null;

								// Check whether items are set in packages or not, current has been
								if ( ! empty( $first_item_in_package ) ) {

									$products_in_package = null;
									$product_quantity    = array();
									$products_name       = array();

									foreach ( $package['Package']['items'] as $product ) {

										$product_quantity[ $product->get_id() ] = isset( $product_quantity[ $product->get_id() ] ) ? ( $product_quantity[ $product->get_id() ] + 1 ) : 1;
										$products_name[ $product->get_id() ]    = $product->get_name();
									}

									foreach ( $products_name as $product_id => $product_name ) {

										if ( ! empty( $products_in_package ) ) {

											$next_product_in_package = '<a style ="text-decoration:none;" href = "' . admin_url( "post.php?post=$product_id&action=edit" ) . '" >' . $product_name . '</a> X ' . $product_quantity[ $product_id ];
											$products_in_package     = $products_in_package . ', ' . $next_product_in_package;
										} else {

											$products_in_package = '<a style ="text-decoration:none;" href = "' . admin_url( "post.php?post=$product_id&action=edit" ) . '" >' . $product_name . '</a> X ' . $product_quantity[ $product_id ];
										}
									}

									echo "<td style='text-align:center; padding: 5px; font-size:16px;'>" . $products_in_package . '</td>';
								}

								echo '</table>';
							}
							?>
								<br />

								<?php

								if ( isset( $ups_label_details['Type'] ) && $ups_label_details['Type'] == 'FREIGHT' ) {

									if ( $custom_tracking && ! empty( $custom_tracking_url ) ) {

										if ( strpos( $custom_tracking_url, '[TRACKING_ID]' ) !== false ) {

											$tracking_url = str_replace( '[TRACKING_ID]', $shipmentId, $custom_tracking_url );
										} else {

											$tracking_url = $custom_tracking_url . $shipmentId;
										}
										?>
										<strong>
											<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
										</strong>
										<a href="<?php echo $tracking_url; ?>" target="_blank">
											<?php echo $shipmentId; ?>
										</a>

										<?php

									} else {

										?>
										<strong>
											<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
										</strong>
										<a href="<?php echo PH_UPS_Config::PH_UPS_TRACKING_URL . $shipmentId; ?>" target="_blank">
											<?php echo $shipmentId; ?>
										</a>

										<?php

									}
								} elseif ( $custom_tracking && ! empty( $custom_tracking_url ) ) {

									if ( strpos( $custom_tracking_url, '[TRACKING_ID]' ) !== false ) {

										$tracking_url = str_replace( '[TRACKING_ID]', $ups_label_details['TrackingNumber'], $custom_tracking_url );
									} else {

										$tracking_url = $custom_tracking_url . $ups_label_details['TrackingNumber'];
									}
									?>
									<strong>
										<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
									</strong>
									<a href="<?php echo $tracking_url; ?>" target="_blank">
										<?php echo $ups_label_details['TrackingNumber']; ?>
									</a>

									<?php

								} else {

									?>
									<strong>
										<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
									</strong>
									<a href="<?php echo PH_UPS_Config::PH_UPS_TRACKING_URL . $ups_label_details['TrackingNumber']; ?>" target="_blank">
										<?php echo $ups_label_details['TrackingNumber']; ?>
									</a>
									<?php

								}

								?>

								<br />

								<a style="margin-top: 7px" class="button button-primary tips" href="<?php echo $download_url; ?>" data-tip="<?php echo __( 'Print Label ' . $post_fix_label, 'ups-woocommerce-shipping' ); ?>" target="<?php echo $target_val; ?>">
									<?php echo __( 'Print Label ' . $post_fix_label, 'ups-woocommerce-shipping' ); ?>
								</a>

								<br /><br />

								<?php

								// Return Label Link
								if ( isset( $created_shipments_details['return'] ) && ! empty( $created_shipments_details['return'] ) ) {

									// Only one return label is considered now
									$return_shipment_id = current( array_keys( $created_shipments_details['return'] ) );

									$ups_return_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_return_label_details_array' );

									// Check for return label accepted data
									if ( is_array( $ups_return_label_details_array ) && isset( $ups_return_label_details_array[ $return_shipment_id ] ) ) {

										$ups_return_label_details = $ups_return_label_details_array[ $return_shipment_id ];

										if ( is_array( $ups_return_label_details ) && isset( $ups_return_label_details[ $index_for_label_count ] ) && ! empty( $ups_return_label_details[ $index_for_label_count ] ) ) {

											$is_forward_label = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_shipment_label_type_as_forward' );

											if ( $is_forward_label == 'forward' ) {

												$ups_return_label_detail = $ups_return_label_details[ $index_for_label_count ];
												$label_index             = $index_for_label_count;
												$return_download_url     = admin_url( '/?wf_ups_print_label=' . base64_encode( $return_shipment_id . '|' . $order_id . '|' . $label_extn_code . '|' . $label_index . '|return' ) . '&nocache=' . time() );

												if ( $custom_tracking && ! empty( $custom_tracking_url ) ) {

													if ( strpos( $custom_tracking_url, '[TRACKING_ID]' ) !== false ) {

														$tracking_url = str_replace( '[TRACKING_ID]', $ups_return_label_detail['TrackingNumber'], $custom_tracking_url );
													} else {

														$tracking_url = $custom_tracking_url . $ups_return_label_detail['TrackingNumber'];
													}

													?>

													<strong>
														<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
													</strong>

													<a href="<?php echo $tracking_url; ?>" target="_blank">
														<?php echo $ups_return_label_detail['TrackingNumber']; ?>
													</a>

													<br />
													<?php

												} else {

													?>
													<strong>
														<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
													</strong>

													<a href="<?php echo PH_UPS_Config::PH_UPS_TRACKING_URL . $ups_return_label_detail['TrackingNumber']; ?>" target="_blank">
														<?php echo $ups_return_label_detail['TrackingNumber']; ?>
													</a>

													<br />
													<?php

												}

												?>

												<a class="button button-primary tips" href="<?php echo $return_download_url; ?>" data-tip="<?php echo __( 'Print Return Label ' . $post_fix_label, 'ups-woocommerce-shipping' ); ?>" target="<?php echo $target_val; ?>">
													<?php __( 'Print Return Label ' . $post_fix_label, 'ups-woocommerce-shipping' ); ?>
												</a>
												<hr style="border-color:#0074a2">

												<?php

											} else {

												$ups_return_label_detail = $ups_return_label_details[ $index_for_label_count ];

												$return_download_url = admin_url( '/?wf_ups_print_label=' . base64_encode( $return_shipment_id . '|' . $order_id . '|' . $label_extn_code . '|' . $index_for_label_count . '|return' ) . '&nocache=' . time() );

												if ( $custom_tracking && ! empty( $custom_tracking_url ) ) {

													if ( strpos( $custom_tracking_url, '[TRACKING_ID]' ) !== false ) {

														$tracking_url = str_replace( '[TRACKING_ID]', $ups_return_label_detail['TrackingNumber'], $custom_tracking_url );
													} else {

														$tracking_url = $custom_tracking_url . $ups_return_label_detail['TrackingNumber'];
													}
													?>

													<strong>
														<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
													</strong>

													<a href="<?php echo $tracking_url; ?>" target="_blank">
														<?php echo $ups_return_label_detail['TrackingNumber']; ?>
													</a>

													<br />
													<?php

												} else {

													?>
													<strong>
														<?php _e( 'Tracking No: ', 'ups-woocommerce-shipping' ); ?>
													</strong>

													<a href="<?php echo PH_UPS_Config::PH_UPS_TRACKING_URL . $ups_return_label_detail['TrackingNumber']; ?>" target="_blank">
														<?php echo $ups_return_label_detail['TrackingNumber']; ?>
													</a>

													<br />

													<?php
												}
												?>

												<a class="button button-primary tips" href="<?php echo $return_download_url; ?>" data-tip="<?php __( 'Print Return Label ' . $post_fix_label, 'ups-woocommerce-shipping' ); ?>" target="<?php echo $target_val; ?>">
													<?php _e( 'Print Return Label ' . $post_fix_label, 'ups-woocommerce-shipping' ); ?>
												</a>

												<hr style="border-color:#0074a2">
												<?php
											}

											++$index_for_label_count;
										}
									}
								}

								// EOF Return Label Link
								$index = $index + 1;
						}
					}

					if ( isset( $ups_commercial_invoice_details[ $shipmentId ] ) ) {

						if ( $this->nafta_co_form || $this->eei_data ) {

							echo '<a class="button button-primary tips" target="' . $target_val . '" href="' . admin_url( '/?wf_ups_print_commercial_invoice=' . base64_encode( $order_id . '|' . $shipmentId ) . '&nocache=' . time() ) . '" data-tip="' . __( 'Downloads International Forms', 'ups-woocommerce-shipping' ) . '">' . __( 'International Forms', 'ups-woocommerce-shipping' ) . '</a><br><br>';
						} else {

							echo '<a class="button button-primary tips" target="' . $target_val . '" href="' . admin_url( '/?wf_ups_print_commercial_invoice=' . base64_encode( $order_id . '|' . $shipmentId ) . '&nocache=' . time() ) . '" data-tip="' . __( 'Downloads Commercial Invoice', 'ups-woocommerce-shipping' ) . '">' . __( 'Commercial Invoice', 'ups-woocommerce-shipping' ) . '</a><br><br>';
						}
					}

					if ( $this->dangerous_goods_manifest && isset( $ups_dangerous_goods_manifest_data[ $shipmentId ] ) && ! empty( $ups_dangerous_goods_manifest_data[ $shipmentId ] ) && $ups_dangerous_goods_manifest_required ) {

						echo '<a class="button button-primary tips" target="_blank" href="' . admin_url( '/?ph_ups_dgm=' . base64_encode( $order_id . '|' . $shipmentId ) . '&nocache=' . time() ) . '" data-tip="' . __( 'Downloads UPS Dangerous Goods Manifest', 'ups-woocommerce-shipping' ) . '">' . __( 'Dangerous Goods Manifest', 'ups-woocommerce-shipping' ) . '</a><br><br>';
					}

						// PDS-129
					if ( $this->dangerous_goods_signatoryinfo && isset( $ups_dangerous_goods_image[ $shipmentId ] ) && ! empty( $ups_dangerous_goods_image[ $shipmentId ] ) ) {

						echo '<a class="button button-primary tips" target="_blank" href="' . admin_url( '/?ph_ups_dangerous_goods_signatoryinfo=' . base64_encode( $order_id . '|' . $shipmentId ) . '&nocache=' . time() ) . '" data-tip="' . __( 'Downloads UPS Dangerous Goods Signatoryinfo', 'ups-woocommerce-shipping' ) . '">' . __( 'Dangerous Goods Signatory Info', 'ups-woocommerce-shipping' ) . '</a><br><br>';
					}

					if ( isset( $ups_control_log_receipt[ $shipmentId ] ) ) {

						echo '<a class="button button-primary tips" target="_blank" href="' . admin_url( '/?ph_ups_print_control_log_receipt=' . base64_encode( $order_id . '|' . $shipmentId ) . '&nocache=' . time() ) . '" data-tip="' . __( 'Print Control Log Receipt', 'ups-woocommerce-shipping' ) . '">' . __( 'Control Log Receipt', 'ups-woocommerce-shipping' ) . '</a><br><br>';
					}

					if ( isset( $created_shipments_details['return'] ) && ! empty( $created_shipments_details['return'] ) ) {

						$return_shipment_id = current( array_keys( $created_shipments_details['return'] ) );

						if ( isset( $ups_return_commercial_invoice_details[ $return_shipment_id ] ) ) {

							echo '<a class="button button-primary tips" target="' . $target_val . '" href="' . admin_url( '/?wf_ups_print_return_commercial_invoice=' . base64_encode( $order_id . '|' . $return_shipment_id ) . '&nocache=' . time() ) . '" data-tip="' . __( 'Print Return Commercial Invoice', 'ups-woocommerce-shipping' ) . '">' . __( 'Return Commercial Invoice', 'ups-woocommerce-shipping' ) . '</a></br><br>';
						}
					}
				}

					// For Create Return label button if it has not been created at the time of label creation
				if ( empty( $created_shipments_details_array[ $shipmentId ]['return'] ) ) {

					$services = base64_encode( PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'xa_ups_generated_label_services' ) );

					echo '<hr style="border-color:#0074a2">';

					$generate_return_label = ! empty( $services ) ? admin_url( "/?xa_generate_return_label=$order_id&service=$services&rt_service=$services" ) : admin_url( "/?xa_generate_return_label=$order_id" );
					echo '<strong>';
					_e( 'Generate Return label : ', 'ups-woocommerce-shipping' );
					echo '</strong>';
					echo '<select name="return_label_service" id="return_label_service">';
					echo '<option value="">' . __( 'Select Your service', 'ups-woocommerce-shipping' ) . '</option>';

					foreach ( $this->ups_services as $key => $value ) {

						if ( $key == 'US48' ) {

							continue;
						}

						echo '<option value="' . $key . '">' . $value . '</option>';
					}

					echo '</select>';
					echo '<a class="button button-primary tips" data-tip="' . __( 'Generate Return Label' ) . '" href ="' . $generate_return_label . '" id="generate_return_label">Generate Return label</a>';
					echo '<hr style="border-color:#0074a2">';

					// To get order id for PH UPS Return Label Type addon
					echo '<input type="hidden" class="ups_order_id" id="ups_order_id" value="' . $order_id . '"/>';
				}
					// End of Create Return label button if it has not been created at the time of label creation

					$void_shipment_url = admin_url( '/?wf_ups_void_shipment=' . base64_encode( $order_id ) . '&nocache=' . time() );

				?>

					<strong>
						<?php _e( 'Cancel the Shipment', 'ups-woocommerce-shipping' ); ?>
					</strong>

					<br />

					<a class="button tips" href="<?php echo $void_shipment_url; ?>" data-tip="<?php _e( 'Void Shipment', 'ups-woocommerce-shipping' ); ?>">
						<?php _e( 'Void Shipment', 'ups-woocommerce-shipping' ); ?>
					</a>

					<hr style="border-color:#0074a2">

					<?php

			} else {

				// If the customer is an OAuth registered customer, fetch shipment tracking IDs.
					$tracking_ids = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_shipment_ids' );

					// Split the tracking IDs into an array.
					$tracking_ids_array = explode( ',', $tracking_ids );

					$ph_trackng_links_elements = '';

					// Loop through each tracking ID and generate the link.
				foreach ( $tracking_ids_array as $tracking_id ) {

					$tracking_link = PH_UPS_Config::PH_UPS_TRACKING_URL . $tracking_id;

					// Add comma separator between tracking links if there are multiple.
					$ph_trackng_links_elements .= ! empty( $ph_trackng_links_elements ) ? ', ' : '';

					// Generate HTML link for each tracking ID
					$ph_trackng_links_elements .= '<a href="' . esc_url( $tracking_link ) . '" target="_blank">' . htmlspecialchars( $tracking_id ) . '</a>';
				}

				?>
						<strong>
						<?php _e( 'Shipment Error', 'ups-woocommerce-shipping' ); ?>
						</strong>
						<p>
							<!-- Display an error message if there was an issue generating shipping labels -->
						<?php _e( 'It looks like there was an issue while generating your shipping labels. Please follow these steps to resolve the issue:', 'ups-woocommerce-shipping' ); ?>
						<ol>
							<!-- Instructions for resolving the shipping label issue -->
							<li><?php _e( 'Copy the tracking details listed below.', 'ups-woocommerce-shipping' ); ?></li>
							<li><?php _e( 'Visit the carrier’s website and cancel the shipping labels for the respective tracking numbers.', 'ups-woocommerce-shipping' ); ?></li>
							<li><?php _e( 'After you’ve completed the cancellation, click the button below to regenerate your shipping labels.', 'ups-woocommerce-shipping' ); ?></li>
							<li><?php _e( 'If the issue persists, contact ', 'ups-woocommerce-shipping' ); ?><a href="https://www.pluginhive.com/support/?utm_source=ups&utm_medium=plugin_settings" target="_blank">support@pluginhive.com</a></li>
						</ol>
						<!-- Display tracking IDs with links to UPS tracking page -->
						<span><?php _e( 'Tracking Ids: ' . $ph_trackng_links_elements, 'ups-woocommerce-shipping' ); ?></span>
						</p>

						<!-- Button to reset and regenerate shipping labels -->
						<a class="button button-primary tips" href="<?php echo admin_url( '/?wf_ups_void_shipment=' . base64_encode( $order_id ) . '&client_reset&nocache=' . time() ); ?>" data-tip="<?php _e( 'Proceed to Regenerate Labels', 'ups-woocommerce-shipping' ); ?>">
						<?php _e( 'Proceed to Regenerate Labels', 'ups-woocommerce-shipping' ); ?>
						</a>
					

					<hr style="border-color:#0074a2">
				<?php
			}
		}
	}

		/**
		 * UPS Document Upload Metabox Content
		 *
		 * @param object $postOrOrderObject
		 */
	public function ph_ups_document_upload_metabox_content( $postOrOrderObject ) {

		$order_object = ( $postOrOrderObject instanceof WP_Post ) ? wc_get_order( $postOrOrderObject->ID ) : $postOrOrderObject;

		if ( ! $order_object instanceof WC_Order ) {

			return;
		}

		$order_id = $order_object->get_id();

		// User Created Form Document Types
		$userCreatedFormDocumentType = array(

			'001' => __( 'Authorization Form', 'ups-woocommerce-shipping' ),
			'002' => __( 'Commercial Invoice', 'ups-woocommerce-shipping' ),
			'003' => __( 'Certificate of Origin', 'ups-woocommerce-shipping' ),
			'004' => __( 'Export Accompanying Document', 'ups-woocommerce-shipping' ),
			'005' => __( 'Export License', 'ups-woocommerce-shipping' ),
			'006' => __( 'Import Permit', 'ups-woocommerce-shipping' ),
			'007' => __( 'One Time NAFTA', 'ups-woocommerce-shipping' ),
			'008' => __( 'Other Document', 'ups-woocommerce-shipping' ),
			'009' => __( 'Power of Attorney', 'ups-woocommerce-shipping' ),
			'010' => __( 'Packing List', 'ups-woocommerce-shipping' ),
			'011' => __( 'SED Document', 'ups-woocommerce-shipping' ),
			'012' => __( "Shipper's Letter of Instruction", 'ups-woocommerce-shipping' ),
			'013' => __( 'Declaration', 'ups-woocommerce-shipping' ),
		);

		?>

			<!-- Hidden inputs to pass data -->

			<input type="hidden" name="order_id" id="order_id" value="<?php echo $order_id; ?>">

			<p>
			<?php echo __( 'Select from the list of UPS additional documents & upload them to UPS electronically.', 'ups-woocommerce-shipping' ); ?>
			</p>

			<span id="ph-loading-spinner"></span>

			<div class='ph-ups-doc-upload-container'>

				<select name="ph_user_created_form_doc_type" id="ph_user_created_form_doc_type">

				<?php

				foreach ( $userCreatedFormDocumentType as $key => $type ) {

					echo '<option value="' . $key . '">' . $type . '</option>';
				}

				?>
				</select>

				<a class="button wf-action-button" id="ph_ups_choose_upload_document"><?php _e( 'Choose File', 'ups-woocommerce-shipping' ); ?></a>

			</div>

			<?php

			$uploadedDocuments = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_upload_document_details' );

			if ( empty( $uploadedDocuments ) ) {
				return;
			}

			echo '<table class="wf-shipment-package-table">';
			echo '<tr>
					<th>' . __( 'Document Type', 'ups-woocommerce-shipping' ) . '</th>
					<th>' . __( 'ID', 'ups-woocommerce-shipping' ) . '</th>
					<th>' . __( 'File Name', 'ups-woocommerce-shipping' ) . '</th>
					<th>' . __( 'Upload Type', 'ups-woocommerce-shipping' ) . '</th>
					<th>' . __( 'Date', 'ups-woocommerce-shipping' ) . '</th>
					<th>' . __( 'Status', 'ups-woocommerce-shipping' ) . '</th>
					<th>' . __( 'Action', 'ups-woocommerce-shipping' ) . '</th>
				</tr>';

			foreach ( $uploadedDocuments as $key => $document ) {

				if ( $document['pushToRepository'] != true ) {

					$deleteDocUrl = admin_url( '/?ph_ups_reupload_document=' . base64_encode( $order_id . '|' . $document['documentID'] ) . '&nocache=' . time() );
					$buttonText   = __( 'Re-upload', 'ups-woocommerce-shipping' );
					$uploadStatus = __( 'Upload Failed', 'ups-woocommerce-shipping' );
				} else {

					$deleteDocUrl = admin_url( '/?ph_ups_delete_document=' . base64_encode( $order_id . '|' . $document['documentID'] ) . '&nocache=' . time() );
					$buttonText   = __( 'Delete', 'ups-woocommerce-shipping' );
					$uploadStatus = __( 'Upload Successful', 'ups-woocommerce-shipping' );
				}

				$uploadStatus = ( isset( $document['isDeleted'] ) && $document['isDeleted'] ) ? __( 'File Deleted', 'ups-woocommerce-shipping' ) : $uploadStatus;
				$wpTimeFormat = get_option( 'time_format' );
				$wpDateFormat = get_option( 'date_format' );

				?>

				<tr>
					<td>
						<?php echo $userCreatedFormDocumentType[ $document['docType'] ]; ?>
					</td>
					<td>
						<?php echo $document['documentID']; ?>
					</td>
					<td>
						<?php echo $document['fileName']; ?>
					</td>
					<td>
						<?php echo $document['uploadType']; ?>
					</td>
					<td>
						<?php echo date( $wpDateFormat . ' ' . $wpTimeFormat, strtotime( $document['uploadDatetime'] ) ); ?>
					</td>
					<td>
						<?php echo $uploadStatus; ?>
					</td>
					<td>
						<?php echo '<a class="button wf-action-button" id="ph_ups_delete_document" href="' . $deleteDocUrl . '">' . $buttonText . '</a>'; ?>
					</td>
				</tr>
				<?php
			}

			echo '</table>';
	}

		// Needed
	private function get_shop_address( $order, $ups_settings ) {

		$shipper_phone_number = isset( $ups_settings['phone_number'] ) ? $ups_settings['phone_number'] : '';
		$attention_name       = isset( $ups_settings['ups_display_name'] ) ? preg_replace( '/&#?[a-z0-9]+;/i', '', $ups_settings['ups_display_name'] ) : '-';
		$company_name         = isset( $ups_settings['ups_user_name'] ) ? preg_replace( '/&#?[a-z0-9]+;/i', '', $ups_settings['ups_user_name'] ) : '-';

		// Multi-Warehouse Addon Support
		$this->set_origin_country_state( $ups_settings );

		// Address standard followed in all xadapter plugins.
		$from_address = array(

			'name'      => $attention_name,
			'company'   => $company_name,
			'phone'     => ( strlen( $shipper_phone_number ) < 10 ) ? '0000000000' : $shipper_phone_number,
			'email'     => isset( $ups_settings['email'] ) ? $ups_settings['email'] : '',
			'address_1' => isset( $ups_settings['origin_addressline'] ) ? $ups_settings['origin_addressline'] : '',
			'address_2' => isset( $ups_settings['origin_addressline_2'] ) ? $ups_settings['origin_addressline_2'] : '',
			'city'      => isset( $ups_settings['origin_city'] ) ? $ups_settings['origin_city'] : '',
			'state'     => $this->origin_state,
			'country'   => $this->origin_country,
			'postcode'  => isset( $ups_settings['origin_postcode'] ) ? $ups_settings['origin_postcode'] : '',
		);

		// Filter for shipping common addon
		return apply_filters( 'wf_filter_label_from_address', $from_address, $this->wf_create_package( $order ) );
	}

		// Needed
	private function get_order_address( $order ) {
		// Address standard followed in all xadapter plugins.
		$billing_address  = $order->get_address( 'billing' );
		$shipping_address = $order->get_address( 'shipping' );

		// Handle the address line one greater than 35 char(UPS Limit)
		$address_line_1_arr = self::divide_sentence_based_on_char_length( $shipping_address['address_1'], 35 );
		$address_line_1     = array_shift( $address_line_1_arr ); // Address Line 1

		// Address Line 2
		if ( ! empty( $address_line_1_arr ) ) {

			$address_line_2 = array_shift( $address_line_1_arr );

			if ( empty( $address_line_1_arr ) ) {

				$address_line_2 = substr( $address_line_2 . ' ' . $shipping_address['address_2'], 0, 35 );
			}
		} else {

			$address_line_2 = substr( $shipping_address['address_2'], 0, 35 );
		}

		$phonenummeta = method_exists( $order, 'get_shipping_phone' ) ? $order->get_shipping_phone() : '';
		$phonenum     = ! empty( $phonenummeta ) ? $phonenummeta : $billing_address['phone'];
		$phone_number = ( strlen( $phonenum ) > 15 ) ? str_replace( ' ', '', $phonenum ) : $phonenum;
		$phone_number = ! empty( $phonenum )
			? $phonenum
			: ( $this->default_recipient_phone && ! empty( $this->default_recipient_phone_num ) ? $this->default_recipient_phone_num : '' );

		return array(

			'name'      => htmlspecialchars( $shipping_address['first_name'] ) . ' ' . htmlspecialchars( $shipping_address['last_name'] ),
			'company'   => ! empty( $shipping_address['company'] ) ? htmlspecialchars( $shipping_address['company'] ) : '-',
			'phone'     => $phone_number,
			'email'     => htmlspecialchars( $billing_address['email'] ),
			'address_1' => htmlspecialchars( $address_line_1 ),
			'address_2' => htmlspecialchars( $address_line_2 ),
			'city'      => htmlspecialchars( $shipping_address['city'] ),
			'state'     => htmlspecialchars( $shipping_address['state'] ),
			'country'   => $shipping_address['country'],
			'postcode'  => $shipping_address['postcode'],
		);
	}

		// Needed
		/**
		 * Get the String divided into multiple sentence based on Character Length of sentence.
		 *
		 * @param $string String String or Sentence on which the Divide has to be applied.
		 * @param $length Length for the new String.
		 * @return array Array of string or sentence of given length
		 */
	public static function divide_sentence_based_on_char_length( $string, $length ) {
		if ( strlen( $string ) <= $length ) {

			return array( $string );
		} else {

			$words_instring = explode( ' ', $string );
			$i              = 0;
			foreach ( $words_instring as $word ) {
				$word = substr( $word, 0, $length );          // To handle the word of length longer than given length
				if ( ! empty( $new_string[ $i ] ) ) {
					$new_length = strlen( $new_string[ $i ] . ' ' . $word );
					if ( $new_length <= $length ) {
						$new_string[ $i ] .= ' ' . $word;
					} else {
						$new_string[ ++$i ] = $word;
					}
				} else {
					$new_string[ $i ] = $word;
				}
			}
			return $new_string;
		}
	}


	private function ph_get_accesspoint_data( $order_details ) {

		$order_details = wc_get_order( $order_details->get_id() );
		$address_field = is_object( $order_details ) ? $order_details->get_meta( '_shipping_accesspoint' ) : '';

		return json_decode( $address_field );
	}

		// Needed
		/**
		 * Get Product from Order Line Item.
		 *
		 * @param array|object $line_item Array in less than woocommerce 3.0 else Object
		 * @return object WC_Product|null|false
		 */
	public function get_product_from_order_item( $line_item ) {
		$product = $line_item->get_product();

		return $product;
	}

		// Needed
	function wf_create_package( $order, $to_address = array() ) {

		// To support Mix and Match Product
		do_action( 'ph_ups_before_get_items_from_order', $order );

		$orderItems = $order->get_items();
		$items      = array();
		$order_id   = $order->get_id();

		foreach ( $orderItems as $orderItem ) {

			$orderItemId       = $orderItem->get_id();
			$refundedItemCount = $order->get_qty_refunded_for_item( $orderItemId );
			$item_id           = $orderItem['variation_id'] ? $orderItem['variation_id'] : $orderItem['product_id'];
			$orderItemQty      = $orderItem['qty'] + $refundedItemCount;

			if ( $orderItemQty <= 0 ) {

				continue;
			}

			if ( empty( $items[ $item_id ] ) ) {

				// $product_data        = wc_get_product( $item_id );
				$product_data = $this->get_product_from_order_item( $orderItem );

				if ( empty( $product_data ) ) {

					$deleted_products[] = $orderItem->get_name();
					continue;
				}

				if ( $product_data->needs_shipping() ) {

					$items[ $item_id ] = array(
						'data'     => $product_data,
						'quantity' => $orderItemQty,
					);
				}
			} else {

				// If a product is in bundle product and it's also ordered individually in same order
				$items[ $item_id ]['quantity'] += $orderItemQty;
			}
		}

		// To support Mix and Match Product
		do_action( 'ph_ups_after_get_items_from_order', $order );

		if ( ! empty( $deleted_products ) && class_exists( 'WC_Admin_Meta_Boxes' ) ) {

			WC_Admin_Meta_Boxes::add_error( __( 'UPS Warning! One or more Ordered Products have been deleted from the Order. Please check these Products- ', 'ups-woocommerce-shipping' ) . implode( ',', $deleted_products ) . '.' );
		}

		// If no items exist in order $items won't be set
		$package['contents'] = isset( $items ) ? apply_filters( 'xa_ups_get_customized_package_items_from_order', $items, $order ) : array();

		$package['destination'] = array(
			'country'   => ! empty( $to_address ) ? $to_address['country'] : $order->get_shipping_country(),
			'state'     => ! empty( $to_address ) ? $to_address['state'] : $order->get_shipping_state(),
			'postcode'  => ! empty( $to_address ) ? $to_address['postcode'] : $order->get_shipping_postcode(),
			'city'      => ! empty( $to_address ) ? $to_address['city'] : $order->get_shipping_city(),
			'address'   => ! empty( $to_address ) ? $to_address['address_1'] : $order->get_shipping_address_1(),
			'address_2' => ! empty( $to_address ) ? $to_address['address_2'] : $order->get_shipping_address_2(),
		);

		// Skip Products
		if ( ! empty( $this->skip_products ) ) {

			$package = $this->skip_products( $package );

			if ( empty( $package['contents'] ) ) {

				return array();
			}
		}

		// Check for Minimum weight and maximum weight
		if ( ! empty( $this->min_weight_limit ) || ! empty( $this->max_weight_limit ) ) {

			$need_shipping = $this->check_min_weight_and_max_weight( $package, $this->min_weight_limit, $this->max_weight_limit );

			if ( ! $need_shipping ) {
				return array();
			}
		}

		$ship_from_address = $this->settings['ship_from_address'];

		$temp = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_shipfrom_address_preference' );

		if ( isset( $_GET['sfap'] ) && ! empty( $_GET['sfap'] ) ) {

			$ship_from_address = $_GET['sfap'];

			PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_shipfrom_address_preference', $_GET['sfap'] );
		} elseif ( ! empty( $temp ) ) {

			$ship_from_address = $temp;
		}

		$packages = apply_filters( 'wf_ups_filter_label_from_packages', array( $package ), $ship_from_address, $order_id );

		return $packages;
	}

		// Needed
		/**
		 * Skip the selected products in settings.
		 *
		 * @param array $package Cart Package.
		 * @param array
		 */
	public function skip_products( $package ) {
		$skipped_products = null;

		foreach ( $package['contents'] as $line_item_key => $line_item ) {

			$line_item_shipping_class = $line_item['data']->get_shipping_class();

			if ( in_array( $line_item_shipping_class, $this->skip_products ) ) {

				$skipped_products[] = $line_item['data']->get_id();

				unset( $package['contents'][ $line_item_key ] );
			}
		}

		if ( $this->debug && ! empty( $skipped_products ) ) {

			$skipped_products = implode( ', ', $skipped_products );

			if ( class_exists( 'WC_Admin_Notices' ) ) {
				WC_Admin_Notices::add_custom_notice( 'ups_skipped_products', __( 'UPS : Skipped Products Id - ', 'ups-woocommerce-shipping' ) . $skipped_products . ' .' );
			}
		}

		if ( $this->debug && ! empty( $skipped_products ) ) {

			$this->admin_diagnostic_report( 'Skipped Products Ids from Label Generation' );
			$this->admin_diagnostic_report( print_r( $skipped_products, 1 ) );
		}

		return $package;
	}

		// Needed
		/**
		 * Check for Order Minimum weight and Maximum weight.
		 *
		 * @param array $package Cart Package.
		 * @param float $min_weight_limit Minimum Weight.
		 * @param float $max_weight_limit Maximum Weight.
		 * @return boolean
		 */
	public function check_min_weight_and_max_weight( $package, $min_weight_limit = null, $max_weight_limit = null ) {
		$package_weight = 0;

		foreach ( $package['contents'] as $line_item ) {

			$quantity         = isset( $line_item['quantity'] ) ? $line_item['quantity'] : 1;
			$line_item_weight = ! empty( $line_item['data']->get_weight() ) ? $line_item['data']->get_weight() : 0;

			$package_weight += (float) ( $line_item_weight * $quantity );
		}

		if ( $package_weight < $min_weight_limit || ( ! empty( $max_weight_limit ) && $package_weight > $max_weight_limit ) ) {

			if ( $this->debug ) {

				if ( class_exists( 'WC_Admin_Notices' ) ) {
					WC_Admin_Notices::add_custom_notice( 'ups_package_weight_not_in_range', __( 'UPS Package Generation stopped. - Package Weight is not in range of Minimum and Maximum Weight Limit (Check UPS Plugin Settings).', 'ups-woocommerce-shipping' ) );
				}

				$this->admin_diagnostic_report( 'UPS Package Generation stopped. - Package Weight is not in range of Minimum and Maximum Weight Limit (Check UPS Plugin Settings).' );
			}
			return false;
		}
		return true;
	}



		/**
		 * Find and add additional error message.
		 *
		 * @param string
		 * @return string
		 */
	public function ph_error_notice_handle( $error_code ) {

		if ( ! class_exists( 'PH_UPS_Error_Notice_Handle' ) ) {

			include 'ph-ups-error-notice-handle.php';
		}

		$error_handel = new PH_UPS_Error_Notice_Handle();

		return $error_handel->ph_find_error_additional_info( $error_code );
	}



	private function wf_create_soap_client( $wsdl, $options ) {

		$soapclient = new SoapClient( $wsdl, $options );

		return $soapclient;
	}

	function wf_ups_print_label() {

		if ( ! $this->wf_user_check( isset( $_GET['auto_generate'] ) ? $_GET['auto_generate'] : null ) ) {

			wp_die( esc_html__( "You don't have admin privileges to view this page.", 'ups-woocommerce-shipping' ), '', array( 'back_link' => 1 ) );
		}

		$query_string    = explode( '|', base64_decode( $_GET['wf_ups_print_label'] ) );
		$shipmentId      = $query_string[0];
		$order_id        = $query_string[1];
		$label_extn_code = $query_string[2];
		$index           = $query_string[3];

		// For return label tracking number is not added, passing $shipmentId
		$tracking_number = $query_string[4] == 'return' ? $shipmentId : $query_string[4];

		// Load Settings.
		$settings_helper = new PH_WC_UPS_Settings_Helper();
		$ups_settings    = $settings_helper->settings;

		$label_meta_name = 'ups_label_details_array';

		if ( isset( $query_string[4] ) ) {

			$return = $query_string[4];

			if ( $return == 'return' ) {

				$label_meta_name = 'ups_return_label_details_array';
			}
		}

		// Legacy Label – Non-OAuth.
		$ups_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, $label_meta_name );

		// OAuth Label.
		if ( empty( $ups_label_details_array ) && Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() ) {

			$label_meta_name = 'ups_rest_label_details_array';

			if ( isset( $query_string[4] ) ) {

				$return = $query_string[4];

				if ( $return == 'return' ) {

					$label_meta_name = 'ups_return_label_details_array';
				}
			}

			$ups_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, $label_meta_name );
		}

		if ( empty( $ups_label_details_array ) ) {

			$ups_label_details_array = array();
		}

		$show_label_in_browser = isset( $ups_settings['show_label_in_browser'] ) ? $ups_settings['show_label_in_browser'] : false;
		$label_in_browser_zoom = isset( $ups_settings['label_in_browser_zoom'] ) && ! empty( $ups_settings['label_in_browser_zoom'] ) ? $ups_settings['label_in_browser_zoom'] : '100';
		$rotate_label          = $ups_settings['rotate_label'];
		$label_format          = ! empty( $ups_settings['label_format'] ) ? $ups_settings['label_format'] : null;
		$margin                = isset( $ups_settings['label_margin'] ) && ! empty( $ups_settings['label_margin'] ) ? $ups_settings['label_margin'] : '0';
		$vertical              = isset( $ups_settings['label_vertical_align'] ) && ! empty( $ups_settings['label_vertical_align'] ) ? $ups_settings['label_vertical_align'] : 'center';
		$horizontal            = isset( $ups_settings['label_horizontal_align'] ) && ! empty( $ups_settings['label_horizontal_align'] ) ? $ups_settings['label_horizontal_align'] : 'center';
		$body_css              = 'margin: ' . $margin . 'px; display: flex; flex-direction: column; justify-content: ' . $vertical . ';';
		$div_css               = 'text-align: ' . $horizontal . ';';

		if ( empty( $ups_label_details_array ) ) {

			// @note - this happens when u created label with rest and delete dregistration
			wf_admin_notice::add_notice( 'UPS: Sorry, An unexpected error occurred.' );

			wp_redirect( admin_url( '/post.php?post=' . $order_id . '&action=edit' ) );
			exit;
		}

		$graphic_image = $ups_label_details_array[ $shipmentId ][ $index ]['GraphicImage'];
		$html_image    = ! empty( $ups_label_details_array[ $shipmentId ][ $index ]['HTMLImage'] ) ? base64_decode( $ups_label_details_array[ $shipmentId ][ $index ]['HTMLImage'] ) : null;

		$file_name_without_extension = 'UPS-ShippingLabel-' . 'Label-' . $order_id . '-' . $tracking_number;

		if ( 'GIF' == $label_extn_code && 'gif' == $ups_settings['print_label_type'] ) {

			// Label format laser_8_5_by_11 should be displayed in browser only when API mode is Live
			if ( $show_label_in_browser ) {

				$html_before_image = "<html><body style='" . $body_css . "'><div style='" . $div_css . "'>";
				$html_after_image  = '</div></body></html>';
				$image_style       = "style='max-width: " . $label_in_browser_zoom . "%;'";

				if ( $label_format == 'laser_8_5_by_11' && ! empty( $html_image ) && 'live' === PH_UPS_Config::PH_UPS_PROXY_ENV) {

					// For Mail Innovation Labels, Image Source is different
					if ( ctype_digit( $shipmentId ) && ( strlen( $tracking_number ) > 18 || ctype_digit( $tracking_number ) ) ) {

						$html_image = str_replace( '<IMG SRC="' . $shipmentId . '_1.GIF"', '<IMG SRC="data:image/gif;base64,' . $graphic_image . '"', $html_image );
					} else {

						$html_image = str_replace( '<IMG SRC="./label' . $tracking_number . '.gif"', '<IMG SRC="data:image/gif;base64,' . $graphic_image . '"', $html_image );
					}

					echo $html_image;
				} else {

					$binary_label = base64_decode( chunk_split( $graphic_image ) );
					$final_image  = $binary_label;

					$source = imagecreatefromstring( $final_image );
					// PDS-73
					if ( $rotate_label ) {

						$final_image = imagerotate( $source, 0, 0 );
					} else {

						$final_image = imagerotate( $source, -90, 0 );
					}

					ob_start();
					imagegif( $final_image );
					$contents = ob_get_contents();
					ob_end_clean();

					echo $html_before_image . '<img ' . $image_style . " src='data:image/gif;base64," . base64_encode( $contents ) . "'/>" . $html_after_image;
				}

				exit;
			}

			$binary_label = base64_decode( chunk_split( $graphic_image ) );
			$final_image  = $binary_label;
			$extn_code    = 'gif';

			if ( $extn_code == 'gif' && ! $show_label_in_browser && $label_format == 'laser_8_5_by_11' && 'live' === PH_UPS_Config::PH_UPS_PROXY_ENV ) {

				$upload_dir_details = wp_get_upload_dir();

				if ( class_exists( 'ZipArchive' ) ) {

					$zip           = new ZipArchive();
					$zip_file_name = $upload_dir_details['basedir'] . '/' . $file_name_without_extension . 'zip';

					if ( $zip->open( $zip_file_name, ZipArchive::CREATE ) ) {

						// For Mail Innovation Labels, Image Source is different
						if ( ctype_digit( $shipmentId ) && ( strlen( $tracking_number ) > 18 || ctype_digit( $tracking_number ) ) ) {

							$html_data = str_replace( $shipmentId . '_1', $file_name_without_extension, $html_image );
						} else {

							$html_data = str_replace( 'label' . $tracking_number, $file_name_without_extension, $html_image );
						}

						$zip->addFromString( $file_name_without_extension . '.html', $html_data );
						$zip->addFromString( $file_name_without_extension . '.gif', $final_image );
						$zip->close();

						header( 'Content-Description: File Transfer' );
						header( 'Content-Type: application/zip' );
						header( 'Content-disposition: attachment; filename="' . $file_name_without_extension . '.zip"' );
						readfile( $zip_file_name );
						unlink( $zip_file_name );
					} else {

						_e( 'Unable to Create Zip file. Please check permission of WP-Upload directory.', 'ups-woocommerce-shipping' );
					}
				} else {

					_e( 'Error - Unable to download Shipping Labels<br/> Reason - ZipArchive class is not enabled for your site<br/> Solution - Please contact your Hosting Provider to enable ZipArchive class for your site and try again', 'ups-woocommerce-shipping' );
				}
			} else {

				header( 'Content-Description: File Transfer' );
				header( 'Content-Type: image/' . $extn_code . '' );
				header( 'Content-disposition: attachment; filename="UPS-ShippingLabel-' . 'Label-' . $order_id . '-' . $tracking_number . '.' . $extn_code . '"' );

				if ( ob_get_contents() ) {
					ob_clean();
				}

				flush();
				echo $final_image;
			}

			exit;
		} elseif ( 'PNG' == $label_extn_code && 'png' == $ups_settings['print_label_type'] ) {

			if ( $show_label_in_browser ) {

				$html_before_image = "<html><body style='" . $body_css . "'><div style='" . $div_css . "'>";
				$html_after_image  = '</div></body></html>';
				$image_style       = "style='max-width: " . $label_in_browser_zoom . "%;'";

				if ( $label_format == 'laser_8_5_by_11' && ! empty( $html_image ) && 'live' === PH_UPS_Config::PH_UPS_PROXY_ENV ) {

					// For Mail Innovation Labels, Image Source is different
					if ( ctype_digit( $shipmentId ) && ( strlen( $tracking_number ) > 18 || ctype_digit( $tracking_number ) ) ) {

						$html_image = str_replace( '<IMG SRC="' . $shipmentId . '_1.PNG"', '<IMG SRC="data:image/png;base64,' . $graphic_image . '"', $html_image );
					} else {

						$html_image = str_replace( '<IMG SRC="./label' . $tracking_number . '.png"', '<IMG SRC="data:image/png;base64,' . $graphic_image . '"', $html_image );
					}

					echo $html_image;
				} else {

					$binary_label = base64_decode( chunk_split( $graphic_image ) );
					$final_image  = $binary_label;
					$extn_code    = 'png';

					$source = imagecreatefromstring( $final_image );
					// PDS-73
					if ( $rotate_label ) {
						$final_image = imagerotate( $source, 0, 0 );
					} else {
						$final_image = imagerotate( $source, -90, 0 );
					}

					ob_start();
					imagepng( $final_image );
					$contents = ob_get_contents();
					ob_end_clean();

					echo $html_before_image . '<img ' . $image_style . " src='data:image/png;base64," . base64_encode( $contents ) . "'/>" . $html_after_image;

					imagedestroy( $final_image );
				}

				exit;
			}

			$binary_label = base64_decode( chunk_split( $graphic_image ) );
			$final_image  = $binary_label;
			$extn_code    = 'png';

			if ( $extn_code == 'png' && ! $show_label_in_browser && $label_format == 'laser_8_5_by_11' && 'live' === PH_UPS_Config::PH_UPS_PROXY_ENV ) {

				$upload_dir_details = wp_get_upload_dir();

				if ( class_exists( 'ZipArchive' ) ) {
					$zip           = new ZipArchive();
					$zip_file_name = $upload_dir_details['basedir'] . '/' . $file_name_without_extension . 'zip';

					if ( $zip->open( $zip_file_name, ZipArchive::CREATE ) ) {

						// For Mail Innovation Labels, Image Source is different
						if ( ctype_digit( $shipmentId ) && ( strlen( $tracking_number ) > 18 || ctype_digit( $tracking_number ) ) ) {

							$html_data = str_replace( $shipmentId . '_1', $file_name_without_extension, $html_image );
						} else {

							$html_data = str_replace( 'label' . $tracking_number, $file_name_without_extension, $html_image );
						}

						$zip->addFromString( $file_name_without_extension . '.html', $html_data );
						$zip->addFromString( $file_name_without_extension . '.png', $final_image );
						$zip->close();
						header( 'Content-Description: File Transfer' );
						header( 'Content-Type: application/zip' );
						header( 'Content-disposition: attachment; filename="' . $file_name_without_extension . '.zip"' );
						readfile( $zip_file_name );
						unlink( $zip_file_name );
					} else {

						_e( 'Unable to Create Zip file. Please check permission of WP-Upload directory.', 'ups-woocommerce-shipping' );
					}
				} else {

					_e( 'Error - Unable to download Shipping Labels<br/> Reason - ZipArchive class is not enabled for your site<br/> Solution - Please contact your Hosting Provider to enable ZipArchive class for your site and try again', 'ups-woocommerce-shipping' );
				}
			} else {
				$source = imagecreatefromstring( $final_image );

				// Rotate
				$final_image = imagerotate( $source, -90, 0 );

				if ( $final_image !== false ) {
					header( 'Content-Description: File Transfer' );
					header( 'Content-Type: image/png' );
					header( 'Content-disposition: attachment; filename="UPS-ShippingLabel-' . 'Label-' . $order_id . '-' . $tracking_number . '.' . $extn_code . '"' );

					if ( ob_get_contents() ) {
						ob_clean();
					}

					flush();
					imagepng( $final_image );
					imagedestroy( $final_image );
				} else {
					echo 'An error occurred.';
				}
			}
			exit;
		}

		// ZPL which will be converted to PNG.
		elseif ( 'ZPL' == $label_extn_code && 'zpl' == $ups_settings['print_label_type'] ) {
			$binary_label = base64_decode( chunk_split( $graphic_image ) );

			// By default zpl code returned by UPS has ^POI command, which will invert the label because
			// of some reason. Removing it so that label will not be inverted.
			$zpl_label_inverted = str_replace( '^POI', '', $binary_label );

			$file_name = 'UPS-ShippingLabel-Label-' . $order_id . '-' . $tracking_number . '.zpl';
			$this->wf_generate_document_file( $zpl_label_inverted, $file_name );
			exit;
		} elseif ( 'EPL' == $label_extn_code && 'epl' == $ups_settings['print_label_type'] ) {
			$binary_label = base64_decode( chunk_split( $graphic_image ) );

			$file_name = 'UPS-ShippingLabel-Label-' . $order_id . '-' . $tracking_number . '.epl';
			$this->wf_generate_document_file( $binary_label, $file_name );
			exit;
		} elseif ( 'PDF' == $label_extn_code ) {
			$binary_label = base64_decode( chunk_split( $graphic_image ) );

			$file_name   = 'UPS-BOL-' . $order_id . '-' . $tracking_number . '.pdf';
			$final_image = $binary_label;
			$extn_code   = 'pdf';

			header( 'Content-Description: File Transfer' );
			header( 'Content-Type: image/' . $extn_code . '' );
			header( 'Content-disposition: attachment; filename="' . $file_name . '"' );

			echo $final_image;

			exit;
		}

		if ( $this->debug ) {

			$this->admin_diagnostic_report( "------------------- Label Printing Error #$order_id -------------------" );
			$this->admin_diagnostic_report( 'Label(s) generated using ' . $label_extn_code . ' format.' );
			$this->admin_diagnostic_report( 'Trying print label in ' . $ups_settings['print_label_type'] . ' format.' );
			$this->admin_diagnostic_report( 'Change the Print Label Type setting back to ' . $label_extn_code . ' format and try again.' );
		}

		wf_admin_notice::add_notice( 'Label(s) generated using ' . $label_extn_code . ' format.' );
		wf_admin_notice::add_notice( 'Trying print label in ' . $ups_settings['print_label_type'] . ' format.' );
		wf_admin_notice::add_notice( 'Change the Print Label Type setting back to ' . $label_extn_code . ' format and try again.' );

		wp_redirect( admin_url( '/post.php?post=' . $order_id . '&action=edit' ) );
		exit;
	}

	function wf_ups_print_commercial_invoice() {
		$req_data = explode( '|', base64_decode( $_GET['wf_ups_print_commercial_invoice'] ) );

		$order_id    = $req_data[0];
		$shipment_id = $req_data[1];

		$invoice_details = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_commercial_invoice_details' );

		$graphic_image = $invoice_details[ $shipment_id ]['GraphicImage'];

		$extn_code = $invoice_details[ $shipment_id ]['ImageFormat'];

		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: image/' . $extn_code . '' );
		header( 'Content-disposition: attachment; filename="UPS-Commercial-Invoice-' . $order_id . '.' . $extn_code . '"' );

		echo base64_decode( $graphic_image );
		exit;
	}

		// PDS-129
	function ph_ups_print_dangerous_goods_signatoryinfo() {

		$req_data        = explode( '|', base64_decode( $_GET['ph_ups_dangerous_goods_signatoryinfo'] ) );
		$order_id        = $req_data[0];
		$shipment_id     = $req_data[1];
		$invoice_details = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_dangerous_goods_image' );

		if ( empty( $invoice_details ) ) {

			// Get from custom meta table
			$invoice_details = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_dangerous_goods_image' );
		}

		$graphic_image = $invoice_details[ $shipment_id ]['DGPaperImage'];
		$extn_code     = 'PDF';

		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: image/' . $extn_code . '' );
		header( 'Content-disposition: attachment; filename="UPS-Dangerous-Goods-Paper-' . $order_id . '.' . $extn_code . '"' );

		echo base64_decode( $graphic_image );
		exit;
	}

	function wf_ups_print_return_commercial_invoice() {

		$req_data        = explode( '|', base64_decode( $_GET['wf_ups_print_return_commercial_invoice'] ) );
		$order_id        = $req_data[0];
		$shipment_id     = $req_data[1];
		$invoice_details = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_return_commercial_invoice_details' );
		$graphic_image   = $invoice_details[ $shipment_id ]['GraphicImage'];
		$extn_code       = $invoice_details[ $shipment_id ]['ImageFormat'];

		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: image/' . $extn_code . '' );
		header( 'Content-disposition: attachment; filename="UPS-Return-Commercial-Invoice-' . $order_id . '.' . $extn_code . '"' );

		echo base64_decode( $graphic_image );
		exit;
	}

	function ph_ups_print_control_log_receipt() {

		$req_data    = explode( '|', base64_decode( $_GET['ph_ups_print_control_log_receipt'] ) );
		$order_id    = $req_data[0];
		$shipment_id = $req_data[1];

		$control_log_receipts = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_control_log_receipt' );
		$control_log_receipt  = $control_log_receipts[ $shipment_id ];

		echo '
				<html>
				<head>
				<meta http-equiv="Content-Type" content="text/html; charset=utf-8">
				<script src="https://ajax.googleapis.com/ajax/libs/jquery/3.3.1/jquery.min.js"></script>
				<script>
				$(document).ready(function(){
					$(document).on("click", "#print_all", function(){
						PrintElem("content");
					})
				});
				function PrintElem(elem)
				{
					var mywindow = window.open("", "PRINT", "height=400,width=600");

					mywindow.document.write("<html><head><title>" + document.title  + "</title>");
					mywindow.document.write("</head><body >");
					mywindow.document.write("<style >");
					mywindow.document.write(`
					@page {
						// size: auto;
						// margin :10px;
					}`);
					mywindow.document.write("</style >");

					mywindow.document.write("<h1>" + document.title  + "</h1>");
					mywindow.document.write(document.getElementById("content").innerHTML);
					mywindow.document.write("</body></html>");

					mywindow.document.close(); // necessary for IE >= 10
					mywindow.focus(); // necessary for IE >= 10*/

					mywindow.print();
					mywindow.close();

					return true;
				}
				</script>
				</head>
				<body>
				<style>
				#print_all{
				    text-decoration: none;
				    display: inline-block;
				    width: 75px;
				    margin: 20px auto;
				    background: linear-gradient(#e3647e, #DC143C);
				    text-align: center;
				    color: #fff;
				    padding: 3px 6px;
				    border-radius: 3px;
				    border: 1px solid #e3647e;
				}
				#print_all:hover{
					background: linear-gradient(#e3647e, #dc143c73);
					cursor: pointer;
				}
				</style>
				<div style="text-align: center;padding: 30px;background: #f3f3f3;margin: 0px 10px 10px 10px;">
					<a id="print_all">Print</a><br/>
				</div>
				<div id="content" style="text-align: center;">
				<table align="center">
				<tr><td>';
		echo $control_log_receipt;
		echo '
				<tr><td>
				</table>
				</div>
				</body>
				</html>';

		exit;
	}

	function ph_ups_print_dangerous_goods_manifest() {

		$req_data      = explode( '|', base64_decode( $_GET['ph_ups_dgm'] ) );
		$order_id      = $req_data[0];
		$shipment_id   = $req_data[1];
		$order_object  = wc_get_order( $order_id );
		$manifest_data = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ph_ups_dangerous_goods_manifest_data' );
		$manifest_data = isset( $manifest_data[ $shipment_id ] ) ? $manifest_data[ $shipment_id ] : array();

		$ups_settings = get_option( 'woocommerce_' . WF_UPS_ID . '_settings', null );
		$ups_settings = apply_filters( 'ph_ups_plugin_settings', $ups_settings, $order_object );

		$ups_shipper_number = isset( $ups_settings['shipper_number'] ) ? $ups_settings['shipper_number'] : '';
		$ship_from_address  = isset( $ups_settings['ship_from_address'] ) ? $ups_settings['ship_from_address'] : 'origin_address';

		$temp = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_shipfrom_address_preference' );

		if ( ! empty( $temp ) ) {

			$ship_from_address = $temp;
		}

		$billing_address_preference = $this->get_product_address_preference( $order_object, $ups_settings, false );

		if ( 'billing_address' == $ship_from_address && $billing_address_preference ) {

			$to_address = $this->get_shop_address( $order_object, $ups_settings );
		} else {

			$to_address = $this->get_order_address( $order_object );
		}

		if ( ! class_exists( 'PH_DGM_Table' ) ) {

			include_once 'FPDF/ph_dgm_table.php';
		}

		$pdf = new PH_DGM_Table();

		// Add a new page
		$pdf->AddPage();

		// Add Heading
		// Add a new font - Font Family, Font Style, Font Size
		$pdf->SetFont( 'Helvetica', '', 25 );
		// Prints a cell - Cell Width, Cell Height, Text, Border,  Position, Align
		$pdf->Cell( 0, 15, 'UPS Dangerous Goods Manifest', 0, 0, 'C' );
		// Line Break
		$pdf->Ln( 20 );

		// Add Account Number and Destination Address
		$pdf->SetFont( 'Helvetica', 'U', 14 );
		$pdf->Cell( 0, 10, $ups_shipper_number, 0, 0, 'L' );
		$pdf->Ln();

		$pdf->SetFont( 'Helvetica', '', 14 );
		$pdf->Cell( 0, 6, $to_address['name'], 0, 0, 'L' );
		$pdf->Ln();

		$pdf->Cell( 0, 6, $to_address['address_1'] . ', ' . $to_address['address_2'], 0, 0, 'L' );
		$pdf->Ln();

		$pdf->Cell( 0, 6, $to_address['city'] . ', ' . $to_address['state'] . ' ' . $to_address['postcode'] . ', ' . $to_address['country'], 0, 0, 'L' );
		$pdf->Ln( 15 );

		// Add Dangerous Goods Manifest Data as Table

		// Table Column Headings
		$header = array(
			'Hazardous Materials Description and Quantity',
			'Regulation Set',
			'The Shipment is within the limitations prescribed for:',
			'Emergency Contact Information',
			'Tracking Number',
		);

		// Set Table Heading Font
		$pdf->SetFont( 'Helvetica', 'B', 13 );

		// Table with 5 columns
		$pdf->PH_SetWidths( array( 50, 30, 40, 35, 35 ) );

		$pdf->PH_Row( $header, true );

		// Set Table Body Font
		$pdf->SetFont( 'Helvetica', '', 12 );

		// Set Table Body Alignment
		$pdf->PH_SetAligns( array( 'L', 'C', 'C', 'L', 'L' ) );

		foreach ( $manifest_data as $productId => $hazmatData ) {

			$descAndQuantity    = $hazmatData['commodityId'] . ', ' . strtoupper( $hazmatData['properShippingName'] ) . ', ' . $hazmatData['classDivisionNumber'] . ', ' . $hazmatData['packagingGroupType'] . ', ' . $hazmatData['quantity'] . ' ' . strtoupper( $hazmatData['packagingType'] ) . ' x ' . $hazmatData['productWeight'] . ' ' . strtoupper( $hazmatData['uom'] );
			$regulationSet      = $hazmatData['regulationSet'];
			$transportationMode = $hazmatData['transportationMode'];
			$emergencyNum       = isset( $ups_settings['phone_number'] ) ? $ups_settings['phone_number'] : '';
			$trackingNumber     = $hazmatData['trackingNumber'];

			// Add Table Rows
			$pdf->PH_Row( array( $descAndQuantity, $regulationSet, $transportationMode, $emergencyNum, $trackingNumber ) );
		}

		$pdf->Output( 'I', 'UPS-Dangerous-Goods-Manifest-' . $order_id . '.pdf' );

		die;
	}

	private function wf_generate_document_file( $content, $file_name ) {

		$uploads_dir_info    = wp_upload_dir();
		$file_name_with_path = $uploads_dir_info['basedir'] . $file_name;
		$handle              = fopen( $file_name_with_path, 'w' );

		fwrite( $handle, $content );
		fclose( $handle );

		header( 'Content-Type: application/' . strtolower( pathinfo( $file_name, \PATHINFO_EXTENSION ) ) );
		header( 'Content-Disposition: attachment; filename=' . basename( $file_name ) );
		header( 'Expires: 0' );
		header( 'Cache-Control: must-revalidate' );
		header( 'Pragma: public' );
		header( 'Content-Length: ' . filesize( $file_name_with_path ) );

		if ( ob_get_contents() ) {

			ob_clean();
		}

		flush();
		readfile( $file_name_with_path );
		unlink( $file_name_with_path );

		return;
	}

	/**
	 * User Check function
	 *
	 * @param string $auto_generate
	 * @return boolean
	 */
	function wf_user_check( $auto_generate = null ) {

		$current_minute = (int) date( 'i' );

		if ( ! empty( $auto_generate ) && ( $auto_generate == md5( $current_minute ) || $auto_generate == md5( $current_minute + 1 ) ) ) {

			return true;
		}

		if ( is_admin() ) {

			return true;
		}

		return false;
	}

	public function get_dimension_from_package( $package ) {

		$dimensions = array(
			'Length'       => null,
			'Width'        => null,
			'Height'       => null,
			'Weight'       => null,
			'InsuredValue' => null,
		);

		if ( ! isset( $package['Package'] ) ) {

			return $dimensions;
		}
		if ( isset( $package['Package']['Dimensions'] ) ) {

			$dimensions['Length'] = (string) round( $package['Package']['Dimensions']['Length'], 2 );
			$dimensions['Width']  = (string) round( $package['Package']['Dimensions']['Width'], 2 );
			$dimensions['Height'] = (string) round( $package['Package']['Dimensions']['Height'], 2 );
		}

		$weight = $package['Package']['PackageWeight']['Weight'];

		if ( $package['Package']['PackageWeight']['UnitOfMeasurement']['Code'] == 'OZS' ) {

			// Make weight in pounds
			if ( $this->weight_unit == 'LBS' ) {

				$weight = $weight / 16;

				// To KG
			} else {
				$weight = $weight / 35.274;
			}
		}

		// PackageServiceOptions
		if ( isset( $package['Package']['PackageServiceOptions']['InsuredValue'] ) ) {

			$dimensions['InsuredValue'] = $package['Package']['PackageServiceOptions']['InsuredValue']['MonetaryValue'];
		}

		$dimensions['Weight'] = (string) round( (float) $weight, 2 );

		return $dimensions;
	}

		// Bulk Label Printing
	function init_bulk_printing() {
		// Legacy Option
		// add_action('admin_footer',   array($this, 'ph_ups_add_bulk_actions'));
		// add_action('load-edit.php',  array($this, 'perform_bulk_label_actions'));

		// HPOs Option
		add_filter( 'bulk_actions-woocommerce_page_wc-orders', array( $this, 'ph_ups_add_bulk_actions' ) );
		add_action( 'admin_init', array( $this, 'ph_ups_handle_bulk_actions_hpo_table' ) );

		// Legacy Option
		add_filter( 'bulk_actions-edit-shop_order', array( $this, 'ph_ups_add_bulk_actions' ) );
		add_filter( 'handle_bulk_actions-edit-shop_order', array( $this, 'ph_ups_handle_bulk_actions_post_table' ), 10, 3 );

		// Add Print Label option to Order list page Make Sure Screen Options->Actions is checked
		add_action( 'woocommerce_admin_order_actions_end', array( $this, 'label_printing_buttons' ) );
	}

		/**
		 * Add bulk actions
		 *
		 * @param array $actions
		 * @return array $actions
		 */
	public function ph_ups_add_bulk_actions( $actions ) {
		if ( $this->disble_ups_print_label != 'yes' ) {

			$actions['ups_generate_label']                  = __( 'Generate UPS Label', 'ups-woocommerce-shipping' );
			$actions['xa_ups_print_label_pdf']              = __( 'Print UPS Label (PDF)', 'ups-woocommerce-shipping' );
			$actions['xa_ups_print_label_image']            = __( 'Print UPS Label (Image)', 'ups-woocommerce-shipping' );
			$actions['ph_ups_print_commercial_invoice_pdf'] = __( 'Print UPS Commercial Invoice (PDF)', 'ups-woocommerce-shipping' );
			$actions['ups_void_shipment']                   = __( 'Void UPS Shipment', 'ups-woocommerce-shipping' );
		}

		return $actions;
	}

		/**
		 * Handle Bulk Actions on new screens (HPOS enabled sites)
		 */
	public function ph_ups_handle_bulk_actions_hpo_table() {
		$action    = isset( $_GET['action'] ) && ! empty( $_GET['action'] ) ? $_GET['action'] : '';
		$action    = empty( $action ) ? ( isset( $_GET['action2'] ) && ! empty( $_GET['action2'] ) ? $_GET['action2'] : '' ) : $action;
		$order_ids = isset( $_GET['id'] ) && is_array( $_GET['id'] ) ? $_GET['id'] : array();

		if ( ! empty( $order_ids ) && is_array( $order_ids ) ) {

			$this->perform_bulk_label_actions( $action, $order_ids );
		}
	}

		/**
		 * Handle bulk actions on old screens (Non HPOS sites)
		 *
		 * @param mixed  $redirect_to
		 * @param string $action
		 * @param array  $post_ids
		 * @return mixed $redirect_to
		 */
	public function ph_ups_handle_bulk_actions_post_table( $redirect_to, $action, $post_ids ) {

		if ( ! empty( $post_ids ) && is_array( $post_ids ) ) {

			$this->perform_bulk_label_actions( $action, $post_ids );
		}

		return $redirect_to;
	}

	function perform_bulk_label_actions( $action, $order_ids ) {

		if ( Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() ) {

			if ( ! class_exists( 'PH_Shipping_UPS_Admin_Rest' ) ) {

				include_once 'ups_rest/class-ph-shipping-ups-rest-admin.php';
			}

			$ph_shipping_ups_admin_rest = new PH_Shipping_UPS_Admin_Rest();
		}

		if ( $action == 'ups_generate_label' ) {

			if ( ! Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() ) {

				if ( $this->debug ) {

					$this->admin_diagnostic_report( 'UPS Label generation is unavailable. Please register your UPS account using UPS OAuth in the Registration section to enable this feature.' );
				}

				wp_die(
					wp_kses_post(
						sprintf(
							/* translators: %s: Registration page link */
							__( 'UPS Label generation is unavailable. Please register your UPS account using UPS OAuth in the <a href="%s" target="_blank" rel="noopener noreferrer">Registration</a> section to enable this feature.', 'ups-woocommerce-shipping' ),
							esc_url( admin_url( 'admin.php?page=ph_ups_registration' ) )
						)
					),
					'',
					array( 'back_link' => true )
				);

			}

			foreach ( $order_ids as $order_id ) {

				$ups_order = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_label_details_array' );

				if ( empty( $ups_order ) ) {

					if ( $ph_shipping_ups_admin_rest->ups_confirm_shipment( $order_id ) ) {

						$ph_metadata_handler = $ph_shipping_ups_admin_rest->ups_accept_shipment( $order_id );

						$ph_metadata_handler->ph_save_meta_data();
					}
				}
			}

			wp_redirect( admin_url( '/edit.php?post_type=shop_order' ) );
			exit;

		} elseif ( $action == 'ups_void_shipment' ) {

			if ( ! Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() ) {

				if ( $this->debug ) {

					$this->admin_diagnostic_report( 'UPS Void Shipment functionality is unavailable. Please register your UPS account using UPS OAuth in the Registration section to enable this feature.' );
				}

				wp_die(
					wp_kses_post(
						sprintf(
							/* translators: %s: Registration page link */
							__( 'UPS Void Shipment functionality is unavailable. Please register your UPS account using UPS OAuth in the <a href="%s" target="_blank" rel="noopener noreferrer">Registration</a> section to enable this feature.', 'ups-woocommerce-shipping' ),
							esc_url( admin_url( 'admin.php?page=ph_ups_registration' ) )
						)
					),
					'',
					array( 'back_link' => true )
				);

			}

			foreach ( $order_ids as $order_id ) {

				$ph_shipping_ups_admin_rest->ups_void_shipment( $order_id );
			}

			wp_redirect( admin_url( '/edit.php?post_type=shop_order' ) );
			exit;

			// Bulk label print in PDF format
		} elseif ( $action == 'xa_ups_print_label_pdf' ) {

			$ups_labels = $this->print_labels_in_bulk( $order_ids );

			$this->print_bulk_labels_as_pdf( $ups_labels );

			// Bulk commercial invoice print in PDF format
		} elseif ( $action == 'ph_ups_print_commercial_invoice_pdf' ) {

			$this->print_commercial_invoice_in_bulk( $order_ids );

			// Bulk label print as IMAGE
		} elseif ( $action == 'xa_ups_print_label_image' ) {

			$ups_labels = $this->print_labels_in_bulk( $order_ids );

			$this->print_labels_in_bulk_as_image( $ups_labels );
		}
	}

		/**
		 * Print Labels in Bulk
		 *
		 * @param $order_ids array Array of Order Ids.
		 */
	protected function print_labels_in_bulk( $order_ids ) {
		$count = null;

		foreach ( $order_ids as $order_id ) {

			$ups_labels_arr = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_label_details_array' );

			if ( empty( $ups_labels_arr ) ) {
				$ups_labels_arr = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_rest_label_details_array' );
			}

			// If label is available then only proceed further
			if ( is_array( $ups_labels_arr ) ) {

				foreach ( $ups_labels_arr as $ups_labels ) {

					foreach ( $ups_labels as $ups_label ) {

						if ( strtolower( $ups_label['Code'] ) != 'epl' ) {

							if ( strtolower( $ups_label['Code'] ) == 'zpl' ) {

								$zpl_label = base64_decode( chunk_split( $ups_label['GraphicImage'] ) );

								// By default zpl code returned by UPS has ^POI command, which will invert the label because
								// of some reason. Removing it so that label will not be inverted.
								$zpl_label_inverted = str_replace( '^POI', '', $zpl_label );

								$response = wp_remote_post(
									'http://api.labelary.com/v1/printers/8dpmm/labels/4x8/0/',
									array(
										'timeout' => 70,
										'body'    => $zpl_label_inverted,
									)
								);

								// 250000 Microseconds - 0.25 Seconds. Hit 4 API Requests per Second.
								// Labelary API has a restriction of 5 requests per second per client.
								usleep( 250000 );

								$final_image = $response['body'];        // In PNG format already decoded
							} else {
								$final_image = base64_decode( chunk_split( $ups_label['GraphicImage'] ) );
							}

							$all_ups_labels[] = array(
								'order_id' => $order_id,
								'type'     => strtolower( $ups_label['Code'] ),
								'label'    => $final_image,
							);
							++$count;
						}
					}
				}
			}
		}

		return ! empty( $all_ups_labels ) ? $all_ups_labels : array();
	}

		/**
		 * Print Commercial invoice in Bulk
		 *
		 * @param array $order_ids.
		 */
	public function print_commercial_invoice_in_bulk( $order_ids ) {

		$commercialInvoices = array();
		$count              = 0;

		if ( is_array( $order_ids ) ) {

			foreach ( $order_ids as $order_id ) {

				$invoice_details = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_commercial_invoice_details' );

				if ( is_array( $invoice_details ) ) {

					foreach ( $invoice_details as $shipmentId => $value ) {

						$commercialInvoices[ 'Commercial Invoice-' . $count . $shipmentId ] = $invoice_details[ $shipmentId ]['GraphicImage'];
						++$count;
					}
				}
			}
		}

		if ( ! empty( $commercialInvoices ) ) {

			$this->ph_ups_print_invoice_in_bulk( $commercialInvoices );
		} else {

			wf_admin_notice::add_notice( 'Commercial Invoices are not available for selected orders .' );
		}
	}


		/**
		 * Print Commercial invoice in PDF .
		 *
		 * @param array $commercialInvoices.
		 */
	public function ph_ups_print_invoice_in_bulk( $commercialInvoices ) {

		if ( ! class_exists( 'PhPdfMerger' ) ) {

			require_once 'PDFMerger/PhPdfMerger.php';
		}

		$pdf   = new PhPdfMerger();
		$path  = wp_upload_dir();
		$files = array();

		if ( ! empty( $commercialInvoices ) ) {

			foreach ( $commercialInvoices as $key => $label ) {

				$response    = base64_decode( $label );
				$shipment_id = $key;
				$file        = $path['path'] . "/ShipmentArtifact-$shipment_id.pdf";

				file_put_contents( $file, $response );
				$files[] = $file;
			}

			// Loop through individual files to create a single PDF file
			foreach ( $files as $file ) {

				$pdf->addPDF( $file );
			}

			$pdf->merge( 'download', 'ups-commercial-invoice-' . date( 'Y-m-d' ) . '.pdf' );
		}
	}

		/**
		 * Bulk label print in PDF format. Supports PNG, GIF, ZPL.
		 *
		 * @param $labels array Array of UPS labels.
		 */
	function print_bulk_labels_as_pdf( $labels ) {

		if ( ! class_exists( 'FPDF' ) ) {

			include_once 'FPDF/fpdf.php';
		}

		$count = 0;
		$pdf   = new FPDF();
		$path  = wp_upload_dir();

		if ( ! empty( $labels ) ) {

			$failed_labels = array();

			foreach ( $labels as $label ) {

				// For PNG & GIF
				$width  = -96.7;
				$height = -119;

				// For ZPL
				if ( $label['type'] == 'zpl' ) {

					$width  = -100;
					$height = -110;
				}

				// Can't use same name FPDF limitation of same name
				$file = $path['path'] . "/ups_bulk_image_$count.png";

				if ( $label['type'] == 'gif' ) {

					$file = $path['path'] . "/ups_bulk_image_$count.gif";

					header( 'Content-type: image/jpeg' );

					$label = imagecreatefromstring( $label['label'] );
					$label = imagerotate( $label, -90, 0 );

					imagegif( $label, $path['path'] . "/ups_bulk_image_$count.gif" );

					$file = $path['path'] . "/ups_bulk_image_$count.gif";
				} elseif ( $label['type'] == 'png' ) {

					$file = $path['path'] . "/ups_bulk_image_$count.png";

					header( 'Content-type: image/png' );

					$label = imagecreatefromstring( $label['label'] );
					$label = imagerotate( $label, -90, 0 );

					imagepng( $label, $path['path'] . "/ups_bulk_image_$count.png" );

					$file = $path['path'] . "/ups_bulk_image_$count.png";
				} else {

					file_put_contents( $file, $label['label'] );
				}

				try {

					$pdf->AddPage();
					$pdf->Image( $file, 0, 0, $width, $height );

					++$count;
				} catch ( Exception $e ) {

					$failed_labels[] = $label;
				}

				unlink( $file );
			}

			if ( isset( $failed_labels ) && ! empty( $failed_labels ) ) {

				foreach ( $failed_labels as $label ) {

					// For PNG & GIF
					$width  = -96.7;
					$height = -119;

					// For ZPL
					if ( $label['type'] == 'zpl' ) {
						$width  = -100;
						$height = -110;
					}

					// Can't use same name FPDF limitation of same name
					$file = $path['path'] . "/ups_bulk_image_$count.png";

					if ( $label['type'] == 'gif' ) {

						$file = $path['path'] . "/ups_bulk_image_$count.gif";

						header( 'Content-type: image/jpeg' );

						$label = imagecreatefromstring( $label['label'] );
						$label = imagerotate( $label, -90, 0 );

						imagegif( $label, $path['path'] . "/ups_bulk_image_$count.gif" );

						$file = $path['path'] . "/ups_bulk_image_$count.gif";
					} elseif ( $label['type'] == 'png' ) {

						$file = $path['path'] . "/ups_bulk_image_$count.png";

						header( 'Content-type: image/png' );

						$label = imagecreatefromstring( $label['label'] );
						$label = imagerotate( $label, -90, 0 );

						imagepng( $label, $path['path'] . "/ups_bulk_image_$count.png" );

						$file = $path['path'] . "/ups_bulk_image_$count.png";
					} else {

						file_put_contents( $file, $label['label'] );
					}

					try {

						$pdf->AddPage();
						$pdf->Image( $file, 0, 0, $width, $height );

						++$count;
					} catch ( Exception $e ) {

						wf_admin_notice::add_notice( 'Order #' . $label['order_id'] . ': Sorry. Something went wrong: ' . $e->getMessage() );
					}

					unlink( $file );
				}
			}

			$pdf->Output( 'D', 'UPS-Shipping-Labels-' . date( 'Y-m-d' ) . '.pdf' );
			die;
		}
	}

		/**
		 * Bulk label print in PNG format. Supports only PNG, GIF Format
		 *
		 * @param $ups_labels array Array of shipping labels
		 */
	protected function print_labels_in_bulk_as_image( $shipping_labels ) {

		$ups_settings          = get_option( 'woocommerce_' . WF_UPS_ID . '_settings', null );
		$label_in_browser_zoom = isset( $ups_settings['label_in_browser_zoom'] ) && ! empty( $ups_settings['label_in_browser_zoom'] ) ? $ups_settings['label_in_browser_zoom'] : '90';
		$rotate_label          = isset( $ups_settings['rotate_label'] ) && ! empty( $ups_settings['rotate_label'] ) ? $ups_settings['rotate_label'] : 'no';

		if ( ! empty( $shipping_labels ) ) {

			echo "<html>
					<body style='margin: 0; display: flex; flex-direction: column; justify-content: center;'>
						<div style='text-align: center;'>";

			foreach ( $shipping_labels as $key => $label ) {

				if ( $label['type'] == 'gif' ) {

					$source = imagecreatefromstring( $label['label'] );

					if ( $rotate_label == 'yes' ) {

						$final_image = imagerotate( $source, 0, 0 );
					} else {

						$final_image = imagerotate( $source, -90, 0 );
					}

					ob_start();
					imagegif( $final_image );

					$contents = ob_get_contents();

					ob_end_clean();
				} elseif ( $label['type'] == 'png' ) {

					$source = imagecreatefromstring( $label['label'] );

					if ( $rotate_label == 'yes' ) {

						$final_image = imagerotate( $source, 0, 0 );
					} else {

						$final_image = imagerotate( $source, -90, 0 );
					}

					ob_start();
					imagepng( $final_image );

					$contents = ob_get_contents();

					ob_end_clean();
				} else {

					$contents = $label['label'];
				}

				echo '<div>';

				echo "<img style='max-width: " . $label_in_browser_zoom . "%;' src='data:image/png;base64," . base64_encode( $contents ) . "'/>";

				echo '</div>';
			}

			echo '		</div>
					</body>
				 </html>';

			exit();
		}
	}

	function get_order_label_details( $order_id ) {

		// Legacy Label – Non-OAuth.
		$ups_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_label_details_array' );

		// The label icon will be displayed only if the shipment label is generated using UPS OAuth and your account is actively registered with OAuth.
		if ( Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() && empty( $ups_label_details_array ) ) {

			$ups_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_rest_label_details_array' );
		}

		if ( ! empty( $ups_label_details_array ) && is_array( $ups_label_details_array ) ) {

			return $ups_label_details_array;
		}

		return false;
	}

	public function ph_create_dangerous_goods_manifest( $order_id, $ph_metadata_handler, $created_shipments_details_array ) {

		$ups_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_label_details_array' );
		$packages                = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_wf_ups_stored_packages' );

		$hazmat_products = array();
		$hazmat_package  = array();

		if ( ! empty( $ups_label_details_array ) && is_array( $ups_label_details_array ) ) {

			foreach ( $created_shipments_details_array as $shipmentId => $created_shipments_details ) {

				$hazmat_products = array();

				if ( ! empty( $ups_label_details_array[ $shipmentId ] ) ) {

					foreach ( $ups_label_details_array[ $shipmentId ] as $ups_label_details ) {

						$tracking_number = isset( $ups_label_details['TrackingNumber'] ) ? $ups_label_details['TrackingNumber'] : '';

						if ( is_array( $packages ) ) {

							$package = array_shift( $packages );

							$first_item_in_package = ( isset( $package['Package']['items'] ) && is_array( $package['Package']['items'] ) ) ? current( $package['Package']['items'] ) : null;

							if ( ! empty( $first_item_in_package ) ) {

								foreach ( $package['Package']['items'] as $product ) {

									$product_id     = $product->get_id();
									$product_weight = wc_get_weight( ( ! empty( $product->get_weight() ) ? $product->get_weight() : 0 ), $this->weight_unit );
									$hazmat_product = 'no';

									$product_var_id = '';

									if ( $product->get_parent_id() ) {
										$parent_id      = $product->get_parent_id();
										$product_var_id = $product->get_id();
									} else {
										$parent_id = $product->get_id();
									}

									if ( ! empty( $product_var_id ) ) {

										$hazmat_product  = get_post_meta( $product_var_id, '_ph_ups_hazardous_materials', 1 );
										$hazmat_settings = get_post_meta( $product_var_id, '_ph_ups_hazardous_settings', 1 );
									}

									if ( $hazmat_product != 'yes' && ! empty( $product_id ) ) {

										$hazmat_product  = get_post_meta( $product_id, '_ph_ups_hazardous_materials', 1 );
										$hazmat_settings = get_post_meta( $product_id, '_ph_ups_hazardous_settings', 1 );
									}

									if ( $hazmat_product == 'yes' && ! empty( $hazmat_settings ) && is_array( $hazmat_settings ) ) {

										if ( isset( $hazmat_products[ $product_id ] ) ) {

											++$hazmat_products[ $product_id ]['quantity'];

											if ( $hazmat_products[ $product_id ]['trackingNumber'] != $tracking_number ) {

												$hazmat_products[ $product_id ]['trackingNumber'] .= ', ' . $tracking_number;
											}
											continue;
										}

										$transportationmode = array(
											'01' => 'Highway',
											'02' => 'Ground',
											'03' => 'PAX',
											'04' => 'CAO',
										);

										if ( isset( $hazmat_settings['_ph_ups_hm_transportaion_mode'] ) && array_key_exists( $hazmat_settings['_ph_ups_hm_transportaion_mode'], $transportationmode ) ) {
											$mode = $transportationmode[ $hazmat_settings['_ph_ups_hm_transportaion_mode'] ];
										}

										$idNumber                 = ! empty( $hazmat_settings['_ph_ups_commodity_id'] ) ? $hazmat_settings['_ph_ups_commodity_id'] : '';
										$properShippingName       = ! empty( $hazmat_settings['_ph_ups_shipping_name'] ) ? $hazmat_settings['_ph_ups_shipping_name'] : '';
										$classDivisionNumber      = ! empty( $hazmat_settings['_ph_ups_class_division_no'] ) ? $hazmat_settings['_ph_ups_class_division_no'] : '';
										$packagingGroupType       = ! empty( $hazmat_settings['_ph_ups_package_group_type'] ) ? $hazmat_settings['_ph_ups_package_group_type'] : '';
										$packagingInstructionCode = ! empty( $hazmat_settings['_ph_ups_package_instruction_code'] ) ? $hazmat_settings['_ph_ups_package_instruction_code'] : '';
										$packagingType            = ! empty( $hazmat_settings['_ph_ups_package_type'] ) ? $hazmat_settings['_ph_ups_package_type'] : '';
										$regulationSet            = ! empty( $hazmat_settings['_ph_ups_hm_regulations'] ) ? $hazmat_settings['_ph_ups_hm_regulations'] : '';
										$transportationMode       = $mode;
										$uom                      = ( $this->uom == 'LB' ) ? 'pound' : 'kg';

										$hazmat_products[ $product_id ] = array(
											'productName' => $product->get_name(),
											'productWeight' => $product_weight,
											'trackingNumber' => $tracking_number,
											'commodityId' => $idNumber,
											'properShippingName' => $properShippingName,
											'classDivisionNumber' => $classDivisionNumber,
											'packagingGroupType' => $packagingGroupType,
											'packagingInstructionCode' => $packagingInstructionCode,
											'packagingType' => $packagingType,
											'regulationSet' => $regulationSet,
											'transportationMode' => $transportationMode,
											'uom'         => $uom,
											'quantity'    => 1,
										);
									}
								}
							}
						}
					}
				}
				$hazmat_package[ $shipmentId ] = $hazmat_products;
			}
		}

		if ( ! empty( $hazmat_package ) ) {

			$ph_metadata_handler->ph_update_meta_data( 'ph_ups_dangerous_goods_manifest_required', true );
			$ph_metadata_handler->ph_update_meta_data( 'ph_ups_dangerous_goods_manifest_data', $hazmat_package );
		}

		return $ph_metadata_handler;
	}

	function get_order_label_links( $order_id ) {

		$links = array();

		// Legacy Label – Non-OAuth.
		$created_shipments_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_created_shipments_details_array' );

		// The label icon will be displayed only if the shipment label is generated using UPS OAuth and your account is actively registered with OAuth.
		if ( Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() && empty( $created_shipments_details_array ) ) {

			$created_shipments_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_rest_created_shipments_details_array' );
		}

		if ( ! empty( $created_shipments_details_array ) ) {

			$ups_label_details_array        = $this->get_order_label_details( $order_id );
			$ups_commercial_invoice_details = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_commercial_invoice_details' );

			foreach ( $created_shipments_details_array as $shipmentId => $created_shipments_details ) {

				$index = 0;

				if ( isset( $ups_label_details_array[ $shipmentId ] ) && is_array( $ups_label_details_array[ $shipmentId ] ) ) {

					foreach ( $ups_label_details_array[ $shipmentId ] as $ups_label_details ) {

						$label_extn_code = $ups_label_details['Code'];
						$tracking_number = isset( $ups_label_details['TrackingNumber'] ) ? $ups_label_details['TrackingNumber'] : '';
						$links[]         = admin_url( '/?wf_ups_print_label=' . base64_encode( $shipmentId . '|' . $order_id . '|' . $label_extn_code . '|' . $index . '|' . $tracking_number ) . '&nocache=' . time() );

						// Return Label Link
						if ( isset( $created_shipments_details['return'] ) && ! empty( $created_shipments_details['return'] ) ) {

							// Only one return label is considered now
							$return_shipment_id             = current( array_keys( $created_shipments_details['return'] ) );
							$ups_return_label_details_array = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, 'ups_return_label_details_array' );

							// Check for return label accepted data
							if ( is_array( $ups_return_label_details_array ) && isset( $ups_return_label_details_array[ $return_shipment_id ] ) ) {

								$ups_return_label_details = $ups_return_label_details_array[ $return_shipment_id ];

								if ( is_array( $ups_return_label_details ) ) {

									// As we took only one label so index is zero
									$label_index = 0;
									$links[]     = admin_url( '/?wf_ups_print_label=' . base64_encode( $return_shipment_id . '|' . $order_id . '|' . $label_extn_code . '|' . $label_index . '|return' ) . '&nocache=' . time() );
								}
							}
						}

						$index = $index + 1;
					}
				}

				if ( isset( $ups_commercial_invoice_details[ $shipmentId ] ) ) {

					$links[] = admin_url( '/?wf_ups_print_commercial_invoice=' . base64_encode( $order_id . '|' . $shipmentId ) . '&nocache=' . time() );
				}
			}
		}

		return $links;
	}

	function label_printing_buttons( $order ) {

		$actions                       = array();
		$labels                        = $this->get_order_label_links( $order->get_id() );
		$commercial_invoice_image_link = plugin_dir_url( __DIR__ ) . 'resources/images/ups-commercial-invoice.png';
		$normal_label_image_link       = plugin_dir_url( __DIR__ ) . 'resources/images/ups-logo-16x16.png';

		if ( is_array( $labels ) ) {

			foreach ( $labels as $label_no => $label_link ) {

				$commercial_label = ( strpos( $label_link, 'commercial' ) === false ) ? false : true;

				$actions[ 'print_label' . $label_no ] = array(
					'url'        => $label_link,
					'name'       => $commercial_label ? __( 'Print UPS Commercial Invoice', 'ups-woocommerce-shipping' ) : __( 'Print UPS Label', 'ups-woocommerce-shipping' ),
					'action'     => 'wf-print-label',
					'image_link' => $commercial_label ? $commercial_invoice_image_link : $normal_label_image_link,
				);
			}
		}

		foreach ( $actions as $action ) {

			printf( '<a class="button tips %s" href="%s" data-tip="%s" target="_blank"><img class="wf-print-label-on-order-list-image" src="' . $action['image_link'] . '"></a>', esc_attr( $action['action'] ), esc_url( $action['url'] ), esc_attr( $action['name'] ), esc_attr( $action['name'] ) );
		}
	}


		/**
		 * To calculate the shipping cost on order page.
		 */
	public function wf_ups_generate_packages_rates() {
		if ( ! $this->wf_user_check() ) {

			wp_die( esc_html__( "You don't have admin privileges to view this page.", 'ups-woocommerce-shipping' ), '', array( 'back_link' => 1 ) );
		}

		// New registration method with active plugin license key
		$isNewAndActiveRegistration = false;
		$apiAccessDetails           = false;

		// Check for active plugin license
		if ( ! Ph_UPS_Woo_Shipping_Common::phHasActiveLicense() ) {

			wp_die( esc_html__( 'Please use a valid plugin license to continue using WooCommerce UPS Shipping Plugin with Print Label.', 'ups-woocommerce-shipping' ), '', array( 'back_link' => 1 ) );
		} else {

			$isNewAndActiveRegistration = true;

			$apiAccessDetails = Ph_Ups_Endpoint_Dispatcher::phGetApiAccessDetails();

			// Proceed with calculate shipping only if api details are available
			if ( ! $apiAccessDetails ) {
				exit;
			}
		}

		$order_id            = base64_decode( $_GET['wf_ups_generate_packages_rates'] );
		$order               = wc_get_order( $order_id );
		$ph_metadata_handler = new PH_UPS_WC_Storage_Handler( $order );
		$box_name_arr        = json_decode( stripslashes( html_entity_decode( $_GET['box_name'] ) ) );
		$box_id_arr          = json_decode( stripslashes( html_entity_decode( $_GET['box_id'] ) ) );
		$length_arr          = explode( ',', $_GET['length'] );
		$width_arr           = explode( ',', $_GET['width'] );
		$height_arr          = explode( ',', $_GET['height'] );
		$weight_arr          = explode( ',', $_GET['weight'] );
		$insurance_arr       = explode( ',', $_GET['insurance'] );
		$get_stored_packages = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_wf_ups_stored_packages' );
		$international_delivery_confirmation_applicable = false;
		if ( ! isset( $get_stored_packages[0] ) ) {

			$get_stored_packages = array( $get_stored_packages );
		}

		if ( isset( $_GET['package_key'] ) ) {

			$package_indexes = json_decode( stripslashes( html_entity_decode( $_GET['package_key'] ) ) );

			if ( ! empty( $package_indexes ) && is_array( $package_indexes ) ) {

				$final_packages = array();

				foreach ( $package_indexes as $packages_index ) {

					if ( isset( $get_stored_packages[ $packages_index ] ) ) {

						$final_packages[] = $get_stored_packages[ $packages_index ];
					}
				}

				$get_stored_packages = $final_packages;

				// Update the packages in database
				$ph_metadata_handler->ph_update_meta_data( '_wf_ups_stored_packages', $get_stored_packages );
			}
		}

		$package_data = $get_stored_packages;
		$rates        = array();

		$no_of_package_entered = count( $weight_arr );
		$no_of_packages        = count( $get_stored_packages );

		// Populate extra packages, if entered manual values
		if ( $no_of_package_entered > $no_of_packages ) {

			// Get first package to clone default data
			$package_clone = current( $get_stored_packages );

			for ( $i = $no_of_packages; $i < $no_of_package_entered; $i++ ) {

				$get_stored_packages[ $i ] = array(

					'Package' => array(

						'PackagingType' => array(

							'Code'        => '02',
							'Description' => 'Package/customer supplied',
						),
						'Description'   => 'Rate',
						'PackageWeight' => array(

							'UnitOfMeasurement' => array(

								'Code' => isset( $package_clone['Package']['PackageWeight']['UnitOfMeasurement']['Code'] ) ? $package_clone['Package']['PackageWeight']['UnitOfMeasurement']['Code'] : '',
							),
						),
					),
				);
			}
		}

		$shipping_obj    = new PH_Shipping_UPS_Rest( $order );
		$settings_helper = new PH_WC_UPS_Settings_Helper();
		$ups_settings    = $settings_helper->settings;

		$shipping_address = $order->get_address( 'shipping' );
		$contents_cost    = null;
		$product_quantity = array();
		$content_quantity = array();

		// To support Mix and Match Product
		do_action( 'ph_ups_before_get_items_from_order', $order );

		$order_items = $order->get_items();

		// Get Contents Cost from Order
		foreach ( $order_items as $order_item ) {

			$orderItemId       = $order_item->get_id();
			$refundedItemCount = $order->get_qty_refunded_for_item( $orderItemId );

			$product_quantity['quantity'] = $order_item->get_quantity();
			$product_quantity['quantity'] = $product_quantity['quantity'] + $refundedItemCount;

			if ( $product_quantity['quantity'] <= 0 ) {

				continue;
			}

			$content_quantity[] = $product_quantity;

			$product = $this->get_product_from_order_item( $order_item );

			if ( is_a( $product, 'WC_Product' ) && $product->needs_shipping() ) {

				$contents_cost += (float) ( ( ! empty( $product->get_price() ) ? $product->get_price() : $this->fixedProductPrice ) * $product_quantity['quantity'] );
			}
		}

		// To support Mix and Match Product
		do_action( 'ph_ups_after_get_items_from_order', $order );

		$address_package = array(

			'contents_cost' => $contents_cost,
			'destination'   => array(

				'address'  => $shipping_address['address_1'] . ' ' . $shipping_address['address_2'],
				'country'  => $shipping_address['country'],
				'state'    => $shipping_address['state'],
				'postcode' => $shipping_address['postcode'],
				'city'     => $shipping_address['city'],

			),
		);

		$boxCode = '';

		// Check Hazardous Materials in Package
		$is_hazardous_materials = false;

		foreach ( $get_stored_packages as $package_key => $package ) {

			if ( ! empty( $package ) ) {

				foreach ( $package as $key => $value ) {

					if ( isset( $value['items'] ) && ! empty( $value['items'] ) ) {

						foreach ( $value['items'] as $item_key => $items ) {

							if ( get_post_meta( $items->get_id(), '_ph_ups_hazardous_materials', 1 ) == 'yes' ) {

								$is_hazardous_materials = true;
								break;
							}
						}
					}

					if ( isset( $value['BoxCode'] ) ) {

						$boxCode = $value['BoxCode'];
					}

					$package_data[ $package_key ][ $key ] = array(

						'PackagingType' => array(

							'Code'        => isset( $value['PackagingType']['Code'] ) ? $value['PackagingType']['Code'] : '02',
							'Description' => 'Package/customer supplied',
						),
						'Description'   => 'Rate',
						'PackageWeight' => array(

							'UnitOfMeasurement' => array(

								'Code' => isset( $shipping_obj->weight_unit ) ? $shipping_obj->weight_unit : $ups_settings['weight_unit'],
							),
						),
					);

					if ( isset( $value['box_name'] ) ) {

						$package_data[ $package_key ][ $key ]['box_name'] = $value['box_name'];
					}

					if ( ! empty( $box_id_arr[ $package_key ] ) ) {

						$package_data[ $package_key ][ $key ]['BoxCode'] = $box_id_arr[ $package_key ];
					}

					if ( ! empty( $box_name_arr[ $package_key ] ) ) {

						$package_data[ $package_key ][ $key ]['box_name'] = $box_name_arr[ $package_key ];
					}

					if ( ! empty( $weight_arr[ $package_key ] ) ) {

						$package_data[ $package_key ][ $key ]['PackageWeight']['Weight']                    = $weight_arr[ $package_key ];
						$package_data[ $package_key ][ $key ]['PackageWeight']['UnitOfMeasurement']['Code'] = isset( $shipping_obj->weight_unit ) ? $shipping_obj->weight_unit : $ups_settings['weight_unit'];
					} else {

						wf_admin_notice::add_notice( sprintf( __( 'UPS rate request failed - Weight is missing. Aborting.', 'ups-woocommerce-shipping' ) ), 'error' );
						// Redirect to same order page
						wp_redirect( admin_url( '/post.php?post=' . $order_id . '&action=edit' ) );
						exit;       // To stay on same order page
					}

					if ( ! empty( $length_arr[ $package_key ] ) && ! empty( $width_arr[ $package_key ] ) && ! empty( $height_arr[ $package_key ] ) ) {

						$package_data[ $package_key ][ $key ]['Dimensions'] = array(

							'UnitOfMeasurement' => array( 'Code' => isset( $shipping_obj->dim_unit ) ? $shipping_obj->dim_unit : $ups_settings['dim_unit'] ),
							'Length'            => $length_arr[ $package_key ],
							'Width'             => $width_arr[ $package_key ],
							'Height'            => $height_arr[ $package_key ],
						);
					} else {

						unset( $package_data[ $package_key ][ $key ]['Dimensions'] );
					}

					if ( ! empty( $insurance_arr[ $package_key ] ) ) {

						$package_data[ $package_key ][ $key ]['PackageServiceOptions']['InsuredValue'] = array(
							'CurrencyCode'  => $shipping_obj->get_ups_currency(),
							'MonetaryValue' => $insurance_arr[ $package_key ],
						);
					}

					if ( isset( $value['PackageServiceOptions'] ) && isset( $value['PackageServiceOptions']['DeliveryConfirmation'] ) ) {

						$package_data[ $package_key ][ $key ]['PackageServiceOptions']['DeliveryConfirmation'] = $value['PackageServiceOptions']['DeliveryConfirmation'];
					}

					$package_delivery_signature = PH_UPS_WC_Storage_Handler::ph_get_meta_data( $order_id, '_ph_ups_package_delivery_signature' . $package_key );

					$from_country = $this->origin_country;
					$to_country   = $shipping_address['country'];

					if ( ( ( $from_country == $to_country ) && in_array( $from_country, $this->dc_domestic_countries ) ) || ( ( $from_country == 'US' || $from_country == 'PR' ) && ( $to_country == 'US' || $to_country == 'PR' ) ) ) {
						$international_delivery_confirmation_applicable = false;
					} else {
						$international_delivery_confirmation_applicable = true;
					}

					if ( ( $from_country == $to_country && in_array( $from_country, array( 'US', 'PR', 'CA' ) ) ) || ( in_array( $from_country, array( 'US', 'PR' ) ) && in_array( $to_country, array( 'US', 'PR' ) ) ) ) {

						$edit_order_sig = json_decode( stripslashes( html_entity_decode( $_GET['sig'] ) ) );
						$sig            = 0;

						$ph_metadata_handler->ph_update_meta_data( '_ph_ups_delivery_signature', $edit_order_sig );

						if ( ! empty( $edit_order_sig ) ) {

							if ( isset( $value['items'] ) && ! empty( $value['items'] ) ) {

								$package_signature = PH_WC_UPS_Common_Utils::get_package_signature( $value['items'] );

								if ( $edit_order_sig == 4 ) {
									$sig = $this->ph_delivery_confirmation > $package_signature ? $this->ph_delivery_confirmation : $package_signature;
								} else {
									$sig = $edit_order_sig;
								}
							} elseif ( $edit_order_sig == 4 ) {

								$sig = $this->ph_delivery_confirmation;
							} else {
								$sig = $edit_order_sig;
							}
						}

						$sig = $sig == 3 ? 3 : ( $sig > 0 ? 2 : 0 );

						$ph_delivery_signature = $sig;

						if ( ! empty( $ph_delivery_signature ) ) {

							$package_data[ $package_key ][ $key ]['PackageServiceOptions']['DeliveryConfirmation']['DCISType'] = $ph_delivery_signature;
						} else {
							unset( $package_data[ $package_key ][ $key ]['PackageServiceOptions']['DeliveryConfirmation'] );
						}
					} else {

						unset( $package_data[ $package_key ][ $key ]['PackageServiceOptions']['DeliveryConfirmation'] );
					}

					if ( isset( $value['items'] ) && ! empty( $value['items'] ) ) {

						$package_data[ $package_key ][ $key ]['items'] = $value['items'];
					}

					if ( isset( $value['DirectDeliveryOnlyIndicator'] ) && $value['DirectDeliveryOnlyIndicator'] == 'yes' ) {
						$package_data[ $package_key ][ $key ]['DirectDeliveryOnlyIndicator'] = 'yes';
					}
				}
			}
		}

		if ( $get_stored_packages != $package_data ) {

			// Update the packages in database
			$ph_metadata_handler->ph_update_meta_data( '_wf_ups_stored_packages', $package_data );
		}

		if ( $this->address_validation && in_array( $address_package['destination']['country'], array( 'US', 'PR' ) ) ) {

			if ( ! class_exists( 'Ph_Ups_Address_Validation_Rest' ) ) {

				require_once 'ups_rest/class-ph-ups-rest-address-validation.php';
			}

			$Ph_Ups_Address_Validation_Rest = new Ph_Ups_Address_Validation_Rest( $address_package['destination'], $ups_settings );
			$residential_code               = $Ph_Ups_Address_Validation_Rest->residential_check;

			if ( $residential_code == 2 ) {

				$shipping_obj->settings['residential'] = true;
				$this->residential                     = true;
			}
		}

		$rate_request    = $shipping_obj->get_rate_requests( $package_data, $address_package, '', '', $international_delivery_confirmation_applicable );
		$rates           = $shipping_obj->process_result( $shipping_obj->get_result( $rate_request, '', 0, $order_id, $isNewAndActiveRegistration, $apiAccessDetails ) );
		$custom_services = isset( $shipping_obj->custom_services ) ? $shipping_obj->custom_services : $ups_settings['services'];

		// Get rates for surepost services only
		foreach ( $this->ups_surepost_services as $service_code ) {

			// If surepost service code enabled
			if ( ! empty( $custom_services[ $service_code ]['enabled'] ) ) {

				$rate_requests = $shipping_obj->get_rate_requests( $package_data, $address_package, 'surepost', $service_code, $international_delivery_confirmation_applicable );
				$rates         = array_merge( $rates, $shipping_obj->process_result( $shipping_obj->get_result( $rate_requests, 'surepost', 0, $order_id, $isNewAndActiveRegistration, $apiAccessDetails ), 'surepost') );
			}
		}

		// Saturday Delivery request for Rate Calculation in Order Edit Page.
		if ( isset( $_GET['wf_ups_generate_packages_rates'] ) && isset( $_GET['satdel'] ) ) {

			$ph_metadata_handler->ph_update_meta_data( '_ph_ups_saturday_delivery', $_GET['satdel'] );

			if ( ! empty( $_GET['satdel'] ) && $_GET['satdel'] == 'true' ) {

				$rate_requests = $shipping_obj->get_rate_requests( $package_data, $address_package, 'saturday', '', $international_delivery_confirmation_applicable );

				$rates = array_merge($rates, $shipping_obj->process_result($shipping_obj->get_result($rate_requests, 'saturday', 0, $order_id, $isNewAndActiveRegistration, $apiAccessDetails), 'saturday'));
			}
		}

		// Get rates for GFP
		if ( isset( $shipping_obj->settings['services']['US48']['enabled'] ) && $shipping_obj->settings['services']['US48']['enabled'] ) {

			if ( ! $is_hazardous_materials ) {

				$rate_requests = $shipping_obj->get_rate_requests( $package_data, $address_package, 'GFP', 'US48' );

				$rates = array_merge( $rates, $shipping_obj->process_result( $shipping_obj->get_result( $rate_requests, 'UPS GFP', $key, '' ), 'GFP' ) );

			}
		}

		$simpleRatePackageCount = ! empty( $package_data ) && is_array( $package_data ) ? count( $package_data ) : 1;
		$currentPackage         = current( $package_data );
		$packageWeight          = $currentPackage['Package']['PackageWeight']['Weight'] ?? 0;
		$isSimpleRateBox        = ( isset( $currentPackage['Package']['BoxCode'] ) && array_key_exists( $currentPackage['Package']['BoxCode'], $this->simpleRateBoxCodes ) ) ? true : false;

		if ( $this->upsSimpleRate && ( $simpleRatePackageCount == 1 ) && ( $packageWeight <= 50 ) && $isSimpleRateBox ) {

			$rate_requests       = $shipping_obj->get_rate_requests( $package_data, $address_package, 'simple_rate', '', $international_delivery_confirmation_applicable );
			$simpleRatesreturned = $shipping_obj->process_result( $shipping_obj->get_result( $rate_requests, 'simple rate', 0, $order_id, $isNewAndActiveRegistration, $apiAccessDetails ), 'simple rate' );

			foreach ( $rates as $rates_key => $value ) {

				if ( isset( $simpleRatesreturned[ $rates_key ] ) ) {

					$rates[ $rates_key ] = $simpleRatesreturned[ $rates_key ];
				}
			}
		}

		$ph_metadata_handler->ph_update_meta_data( 'wf_ups_generate_packages_rates_response', $rates );

		$ph_metadata_handler->ph_save_meta_data();

		// Redirect to same order page
		wp_redirect( admin_url( '/post.php?post=' . $order_id . '&action=edit#PH_UPS_Metabox' ) );
		exit;       // To stay on same order page
	}



		// Check for any Product has Origin Address Preference, If 'yes' use Origin Address irrespective of Product and Settings
	public function get_product_address_preference( $order, $ups_settings, $return_label = false ) {

		$billing_address = true;

		if ( $order instanceof WC_Order ) {

			// To support Mix and Match Product
			do_action( 'ph_ups_before_get_items_from_order', $order );

			$order_items = $order->get_items();

			if ( ! empty( $order_items ) ) {

				foreach ( $order_items as  $item_key => $item_values ) {

					$orderItemId       = $item_values->get_id();
					$refundedItemCount = $order->get_qty_refunded_for_item( $orderItemId );

					$orderItemQty = $item_values->get_quantity() + $refundedItemCount;

					if ( $orderItemQty <= 0 ) {

						continue;
					}

					$order_item_id = $item_values->get_variation_id();

					$product_id = wp_get_post_parent_id( $order_item_id );

					if ( empty( $product_id ) ) {

						$product_id = $item_values->get_product_id();
					}

					$default_to_origin = get_post_meta( $product_id, '_ph_ups_product_address_preference', 1 );

					if ( $default_to_origin == 'yes' ) {

						$billing_address = false;
						break;
					}
				}
			}

			// To support Mix and Match Product
			do_action( 'ph_ups_after_get_items_from_order', $order );
		}

		return $billing_address;
	}




	public function admin_diagnostic_report( $data ) {

		if ( function_exists( 'wc_get_logger' ) && $this->debug ) {

			$log = wc_get_logger();
			$log->debug( ( $data ) . PHP_EOL . PHP_EOL, array( 'source' => PH_UPS_DEBUG_LOG_FILE_NAME ) );
		}
	}
}

	new WF_Shipping_UPS_Admin();
