<?php

defined('ABSPATH') || exit;

class Ph_Ups_Endpoint_Dispatcher
{

    /**
     * Plugin settings.
     *
     * @var array
     */
    public $settings;

    /**
     * Constructor.
     */
    public function __construct() {
        add_action( 'ph_ups_log_debug_event', array( $this, 'ph_ups_debug_log_every_fifteen_minutes' ) );
    }

    /**
     * Logs internal endpoint information every fifteen minutes.
     *
     * This function:
     * - Loads plugin settings.
     * - Retrieves the appropriate internal endpoints transient key.
     * - Logs key information and internal endpoints data if debug mode is enabled.
     *
     * @return void
     */
    function ph_ups_debug_log_every_fifteen_minutes() {
        
        $settings_helper = new PH_WC_UPS_Settings_Helper();
        $settings        = $settings_helper->settings;

        // Filter for mutli warehouse - multiple account support.
        $warehouse_id = apply_filters('ph_multi_warehouse_id', 'PRIMARY');

        $internal_endpoints_key = ! empty( $warehouse_id )
            ? "PH_UPS_INTERNAL_ENDPOINTS_{" . PH_UPS_Config::PH_UPS_PROXY_ENV . "}_{$warehouse_id}"
            : "PH_UPS_INTERNAL_ENDPOINTS_{" . PH_UPS_Config::PH_UPS_PROXY_ENV . "}";

        $internalEndpoints = get_transient( $internal_endpoints_key );
    
        Ph_UPS_Woo_Shipping_Common::phAddDebugLog(
            '------------------------------- PH UPS Get Internal Endpoints -------------------------------',
            $settings['debug']
        );
    
        Ph_UPS_Woo_Shipping_Common::phAddDebugLog(
            print_r( $internalEndpoints, true ),
            $settings['debug']
        );
    }

    /**
     * Fetch internal API endpoints from proxy server
     *
     * @param string $authProviderToken
     * @return array $endpoints
     */
    public static function phGetInternalEndpoints($authProviderToken)
    {
        $result     = [];
        $endpoints  = [];

        $upsSettings    = get_option('woocommerce_' . WF_UPS_ID . '_settings', null);
        $debug          = (isset($upsSettings['debug']) && !empty($upsSettings['debug']) && $upsSettings['debug'] == 'yes') ? true : false;

        $carrier_slug = Ph_UPS_Woo_Shipping_Common::ph_is_oauth_registered_customer() ? 'ups-rest' : 'ups';

        // Filter for mutli warehouse - multiple account support.
        $warehouse_id = apply_filters('ph_multi_warehouse_id', 'PRIMARY');

        $internal_endpoints_key = ! empty( $warehouse_id )
            ? "PH_UPS_INTERNAL_ENDPOINTS_{" . PH_UPS_Config::PH_UPS_PROXY_ENV . "}_{$warehouse_id}"
            : "PH_UPS_INTERNAL_ENDPOINTS_{" . PH_UPS_Config::PH_UPS_PROXY_ENV . "}";

        if ( empty( get_transient( $internal_endpoints_key ) ) ) {

            $result = Ph_Ups_Api_Invoker::phCallApi(PH_UPS_Config::PH_UPS_PROXY_API_BASE_URL . PH_UPS_Config::PH_UPS_CARRIER_ENDPOINT . $carrier_slug, $authProviderToken, [], [], 'GET');

            if (!empty($result) && is_array($result) && isset($result['response'])) {

                if (isset($result['response']['code']) && $result['response']['code'] == 200 && isset($result['body'])) {

                    $result = json_decode($result['body'], 1);
                    $endpoints = $result['_links'];

                    // Update the endpoints in transient.
                    set_transient( $internal_endpoints_key, $endpoints, 1800 );

                    Ph_UPS_Woo_Shipping_Common::phAddDebugLog('------------------------------- PH UPS Set Internal Endpoints -------------------------------', $debug);
                    Ph_UPS_Woo_Shipping_Common::phAddDebugLog( print_r($endpoints, true), $debug);

                } else {

                    Ph_UPS_Woo_Shipping_Common::phAddDebugLog('------------------------------- Failed to get Internal Endpoints -------------------------------', $debug);
                    Ph_UPS_Woo_Shipping_Common::phAddDebugLog($result['response']['message'], $debug);
                }
            } else {

                Ph_UPS_Woo_Shipping_Common::phAddDebugLog('------------------------------- Failed to get Internal Endpoints -------------------------------', $debug);
                Ph_UPS_Woo_Shipping_Common::phAddDebugLog($result, $debug);
            }
        }

        return $endpoints;
    }

    /**
     * Retrieve Auth Provider access token and the internal API endpoints
     *
     * @return array $apiAccessDetails
     */
    public static function phGetApiAccessDetails()
    {
        $upsSettings = get_option('woocommerce_' . WF_UPS_ID . '_settings', null);

        $debug   = (isset($upsSettings['debug']) && !empty($upsSettings['debug']) && $upsSettings['debug'] == 'yes') ? true : false;

        // Filter for mutli warehouse - multiple account support.
        $warehouse_id = apply_filters( 'ph_multi_warehouse_id', 'PRIMARY' );

        $auth_token_key         = ! empty( $warehouse_id ) ? "PH_UPS_AUTH_PROVIDER_TOKEN_{$warehouse_id}" : "PH_UPS_AUTH_PROVIDER_TOKEN";
        $internal_endpoints_key = ! empty( $warehouse_id )
            ? "PH_UPS_INTERNAL_ENDPOINTS_{" . PH_UPS_Config::PH_UPS_PROXY_ENV . "}_{$warehouse_id}"
            : "PH_UPS_INTERNAL_ENDPOINTS_{" . PH_UPS_Config::PH_UPS_PROXY_ENV . "}";

        $authProviderToken = get_transient( $auth_token_key );
        $internalEndpoints = get_transient( $internal_endpoints_key );

        if (empty($authProviderToken)) {
            include_once('class-ph-ups-auth-handler.php');

            $authProviderToken = Ph_Ups_Auth_Handler::phGetAuthProviderToken();
        }

        if (!empty($authProviderToken) && empty($internalEndpoints)) {

            $internalEndpoints = Ph_Ups_Endpoint_Dispatcher::phGetInternalEndpoints($authProviderToken);
        }

        if(empty($authProviderToken) || empty($internalEndpoints))
        {
            return false;
        }

        $apiAccessDetails = [
            'token'             => $authProviderToken,
            'internalEndpoints' => $internalEndpoints,
        ];

        return $apiAccessDetails;
    }

    /**
     * Return the proxy endpoint for the given proxy service
     *
     * @param string $service
     * @return string 
     */
    public static function ph_get_enpoint($service = '') {

        $api_access_details = self::phGetApiAccessDetails();

        $internal_endpoints = isset($api_access_details['internalEndpoints']) ? $api_access_details['internalEndpoints'] : '';

        return isset($internal_endpoints[$service]['href']) ? $internal_endpoints[$service]['href'] : '';
    }
}

new Ph_Ups_Endpoint_Dispatcher();